#!/usr/bin/env python3
"""
Convert text file (X Y Z R G B Nx Ny Nz) to E57 format.
Uses R value as intensity and allows manual scanner position specification.
"""

import numpy as np
import argparse
from pathlib import Path
import pye57
from datetime import datetime


def load_point_cloud(filepath):
    """
    Load point cloud from text file with format: X Y Z R G B Nx Ny Nz
    
    Returns:
        points: Nx3 array of XYZ coordinates
        colors: Nx3 array of RGB values (0-255)
        intensity: N array of intensity values (from R channel)
    """
    print(f"Loading point cloud from {filepath}...")
    data = np.loadtxt(filepath)
    
    if data.shape[1] < 6:
        raise ValueError(f"Expected at least 6 columns (X Y Z R G B), got {data.shape[1]}")
    
    points = data[:, 0:3]
    colors = data[:, 3:6]
    
    # RGB values are in 0-1 range, convert to 0-255 for E57
    colors_255 = (colors * 255).astype(np.uint8)
    
    # Use R channel as intensity (keep in 0-1 range)
    intensity = colors[:, 0]
    
    print(f"Loaded {len(points)} points")
    print(f"  X range: [{np.min(points[:, 0]):.3f}, {np.max(points[:, 0]):.3f}]")
    print(f"  Y range: [{np.min(points[:, 1]):.3f}, {np.max(points[:, 1]):.3f}]")
    print(f"  Z range: [{np.min(points[:, 2]):.3f}, {np.max(points[:, 2]):.3f}]")
    print(f"  RGB range: [{np.min(colors):.3f}, {np.max(colors):.3f}] (0-1 format)")
    print(f"  Intensity range: [{np.min(intensity):.3f}, {np.max(intensity):.3f}]")
    
    return points, colors_255, intensity


def save_to_e57(points, colors, intensity, output_path, scanner_position, 
                scanner_name="Manual Scanner", guid=None):
    """
    Save point cloud data to E57 format.
    
    IMPORTANT: E57 files store points in scanner-local coordinates, then the pose
    transform converts them to world coordinates when reading. Since our points are
    already in world coordinates, we must apply the INVERSE transform before writing.
    
    Args:
        points: Nx3 array of XYZ coordinates (in world coordinates)
        colors: Nx3 array of RGB values (0-255)
        intensity: N array of intensity values (0-1)
        output_path: Path to save E57 file
        scanner_position: Tuple of (x, y, z) for scanner origin in world coordinates
        scanner_name: Name of the scanner
        guid: Optional GUID for the scan
    """
    import pye57.libe57 as libe57
    
    print(f"\nSaving to E57 format: {output_path}")
    print(f"Scanner position (world coords): [{scanner_position[0]:.3f}, {scanner_position[1]:.3f}, {scanner_position[2]:.3f}]")
    
    scanner_x, scanner_y, scanner_z = scanner_position
    
    # CRITICAL: Transform points from world coordinates to scanner-local coordinates
    # Scanner-local coords = World coords - Scanner position
    # When software reads the E57 and applies the pose transform (+ scanner position),
    # the points will be back in world coordinates
    print(f"Converting points from world to scanner-local coordinates...")
    points_local = points.copy()
    points_local[:, 0] -= scanner_x
    points_local[:, 1] -= scanner_y
    points_local[:, 2] -= scanner_z
    
    print(f"  World coords range: X[{np.min(points[:, 0]):.3f}, {np.max(points[:, 0]):.3f}]")
    print(f"  Local coords range: X[{np.min(points_local[:, 0]):.3f}, {np.max(points_local[:, 0]):.3f}]")
    
    # Create E57 file
    e57 = pye57.E57(output_path, mode='w')
    
    # Prepare data dictionary with scanner-local coordinates
    data = {
        "cartesianX": points_local[:, 0].astype(np.float64),
        "cartesianY": points_local[:, 1].astype(np.float64),
        "cartesianZ": points_local[:, 2].astype(np.float64),
    }
    
    # Add intensity if available (already in 0-1 range from R channel)
    if intensity is not None:
        data["intensity"] = intensity.astype(np.float64)
    
    # Add color data (already converted to 0-255)
    if colors is not None:
        data["colorRed"] = colors[:, 0]
        data["colorGreen"] = colors[:, 1]
        data["colorBlue"] = colors[:, 2]
    
    print(f"Writing {len(points)} points to E57...")
    
    # Convert scanner position to numpy array
    translation = np.array([scanner_x, scanner_y, scanner_z], dtype=np.float64)
    rotation = np.array([1, 0, 0, 0], dtype=np.float64)  # Identity quaternion
    
    # Write with translation
    # Points are in scanner-local coords, pose will transform them back to world coords
    e57.write_scan_raw(
        data,
        name=scanner_name,
        translation=translation,
        rotation=rotation
    )
    
    e57.close()
    
    print(f"\n✓ Successfully saved E57 file")
    print(f"  Points stored in: scanner-local coordinates")
    print(f"  Pose will transform to: world coordinates")
    print(f"  Scanner position: [{scanner_x:.3f}, {scanner_y:.3f}, {scanner_z:.3f}]")
    
    # Verify the file
    verify_e57(output_path, points, scanner_position)


def verify_e57(filepath, original_points=None, scanner_position=None):
    """Verify the E57 file was created correctly and display info."""
    import pye57.libe57 as libe57
    
    print(f"\nVerifying E57 file...")
    
    try:
        e57 = pye57.E57(filepath, mode='r')
        
        print(f"File format: E57")
        print(f"Number of scans: {e57.scan_count}")
        
        if e57.scan_count > 0:
            print(f"\nScan 0 info:")
            
            # Access the scan structure
            try:
                root = e57.image_file.root()
                scan = root["data3D"][0]
                
                # Get name
                if scan.isDefined("name"):
                    name_node = scan["name"]
                    print(f"  Name: {name_node.value()}")
                
                # Get pose/translation
                if scan.isDefined("pose"):
                    pose = scan["pose"]
                    if pose.isDefined("translation"):
                        trans = pose["translation"]
                        
                        x = trans["x"].value()
                        y = trans["y"].value()
                        z = trans["z"].value()
                        
                        print(f"  Scanner Position (pose translation):")
                        print(f"    X: {x:.3f}")
                        print(f"    Y: {y:.3f}")
                        print(f"    Z: {z:.3f}")
            except Exception as e:
                print(f"  Could not read pose details: {e}")
            
            # Read points with and without transform
            try:
                # Read without transform (scanner-local coords)
                scan_local = e57.read_scan(0, ignore_missing_fields=True, transform=False)
                print(f"\n  Point count: {len(scan_local['cartesianX'])}")
                print(f"  Scanner-local coords (as stored):")
                print(f"    X range: [{np.min(scan_local['cartesianX']):.3f}, {np.max(scan_local['cartesianX']):.3f}]")
                
                # Read with transform (world coords)
                scan_world = e57.read_scan(0, ignore_missing_fields=True, transform=True)
                print(f"  World coords (after pose transform):")
                print(f"    X range: [{np.min(scan_world['cartesianX']):.3f}, {np.max(scan_world['cartesianX']):.3f}]")
                
                # Verify against original if provided
                if original_points is not None:
                    orig_x_min = np.min(original_points[:, 0])
                    orig_x_max = np.max(original_points[:, 0])
                    world_x_min = np.min(scan_world['cartesianX'])
                    world_x_max = np.max(scan_world['cartesianX'])
                    
                    if abs(orig_x_min - world_x_min) < 0.01 and abs(orig_x_max - world_x_max) < 0.01:
                        print(f"  ✓ World coords match original points!")
                    else:
                        print(f"  ⚠ World coords don't match original:")
                        print(f"    Original X: [{orig_x_min:.3f}, {orig_x_max:.3f}]")
                        print(f"    Read back X: [{world_x_min:.3f}, {world_x_max:.3f}]")
                
            except Exception as e:
                print(f"  Could not read scan data: {e}")
        
        e57.close()
        print("\n✓ Verification complete!")
        
    except Exception as e:
        print(f"Error verifying E57 file: {e}")
        import traceback
        traceback.print_exc()


def main():
    parser = argparse.ArgumentParser(
        description="Convert text file to E57 format with manual scanner position",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Format: Input text file should have columns: X Y Z R G B Nx Ny Nz
        RGB values should be in 0-1 range (will be converted to 0-255 for E57)
        Normals (Nx Ny Nz) will be discarded
        R value (0-1) will be used as intensity

Examples:
  # Basic conversion with scanner at origin
  python txt_to_e57.py scan.txt scan.e57 --scanner 0 0 0
  
  # Specify scanner position (NO COMMAS - use spaces!)
  python txt_to_e57.py scan.txt scan.e57 --scanner -56.086 -9.308 1.191
  
  # Add scanner name
  python txt_to_e57.py scan.txt scan.e57 --scanner -56.086 -9.308 1.191 --name "Faro"
        """
    )
    
    parser.add_argument("input_file", type=str, 
                        help="Input text file (X Y Z R G B Nx Ny Nz)")
    parser.add_argument("output_file", type=str, 
                        help="Output E57 file")
    parser.add_argument("--scanner", type=float, nargs=3, required=True,
                        metavar=("X", "Y", "Z"),
                        help="Scanner position coordinates (required)")
    parser.add_argument("--name", type=str, default="Manual Scanner",
                        help="Scanner name (default: 'Manual Scanner')")
    parser.add_argument("--guid", type=str, default=None,
                        help="Optional GUID for the scan")
    
    args = parser.parse_args()
    
    # Validate input file exists
    input_path = Path(args.input_file)
    if not input_path.exists():
        print(f"Error: Input file not found: {args.input_file}")
        return 1
    
    # Ensure output has .e57 extension
    output_path = Path(args.output_file)
    if output_path.suffix.lower() != '.e57':
        output_path = output_path.with_suffix('.e57')
        print(f"Adding .e57 extension: {output_path}")
    
    # Load point cloud
    try:
        points, colors, intensity = load_point_cloud(args.input_file)
    except Exception as e:
        print(f"Error loading point cloud: {e}")
        return 1
    
    # Convert to E57
    try:
        scanner_position = tuple(args.scanner)
        save_to_e57(
            points, 
            colors, 
            intensity, 
            str(output_path),
            scanner_position,
            scanner_name=args.name,
            guid=args.guid
        )
    except Exception as e:
        print(f"Error converting to E57: {e}")
        import traceback
        traceback.print_exc()
        return 1
    
    print("\nConversion complete!")
    return 0


if __name__ == "__main__":
    exit(main())
