import sys
import csv
import os.path
import datetime
from copy import deepcopy

import survey_metadata as sm

from office365.runtime.auth.authentication_context import AuthenticationContext
from office365.runtime.client_request import ClientRequest
from office365.sharepoint.client_context import ClientContext
from office365.runtime.auth.client_credential import ClientCredential

import survey_metadata as sm

CONFIG_FILE_LOCATION = "config_files/sharepoint_upload.config"
SHAREPOINT_URL = "https://oh3d.sharepoint.com/sites/OH3D_ingest_form/"

SURVEY_TEMPLATE_PATH = "CSV_files/OH_Metadata_Survey_Template.csv"

READY_STATUS = "Ready to be Published"

def get_user_credentials() -> tuple([str, str]):
    if os.path.isfile(CONFIG_FILE_LOCATION):
        client_id, client_secret = read_config_file()

        if not is_valid_credentials(client_id, client_secret):
            print("Config file credentials are invalid.")
            client_id, client_secret = prompt_user_for_credentials()
    else:
        client_id, client_secret = prompt_user_for_credentials()
        
    return client_id, client_secret    


def read_config_file() -> tuple[str, str]:
    """
    Parses the config file at CONFIG_FILE_LOCATION and returns the found username and password in a tuple. Will return
    blank values for the client id and client secret if it cannot find them. These credentials are not guaranteed to work.

    :return: A tuple, representing the credentials stored in the config file, in the following format: 
             (config_client_id, config_client_secret)
    """
    print("Reading OH3D Sharepoint Credentials from " + CONFIG_FILE_LOCATION + ".")

    config_file = open(CONFIG_FILE_LOCATION, 'r')

    config_client_id = ""
    config_client_secret = ""
    for line in config_file.readlines():
        line_words = line.split(": ")

        if line_words[0] == "Client ID":
            config_client_id = line_words[1][:-1]
        elif line_words[0] == "Client Secret":
            config_client_secret = line_words[1]
    return config_client_id, config_client_secret


def prompt_user_for_credentials() -> tuple[str, str]:
    """
    Prompts the user to manually input a client id and then client secret. Will test the credentials and prompt the user to
    try again if credentials are not correct. The returned credentials are guaranteed to be valid.

    :return: A tuple, representing credentials provided by the user, which are guaranteed to be correct
    """
    while True:
        client_id_input = input("Enter OH3D Sharepoint Client ID: ")
        client_secret_input = input("Enter OH3D Sharepoint Client Secret: ")

        if is_valid_credentials(client_id_input, client_secret_input):
            save_credentials(client_id_input, client_secret_input)

            return client_id_input, client_secret_input

        print("Entered credentials are invalid. Please try again.")


def is_valid_credentials(client_id_in: str, client_secret_in: str) -> bool:
    """
    Tests the given username and password and returns whether the credentials are valid by attempting to connect to the
    database with them.

    :param client_id_in: The client id of the credentials to test
    :param client_secret_in: The client secret of the credentials to test
    :return: Whether the credentials are valid
    """
    try:
        client_credentials = ClientCredential(client_id_in, client_secret_in)
        ctx = ClientContext(SHAREPOINT_URL).with_credentials(client_credentials)

        list_object = ctx.web.lists.get_by_title("Projects")
        items = list_object.get_items()
        ctx.load(items)
        ctx.execute_query()

        return True
    except NameError:
        return False


def save_credentials(client_id_to_save: str, client_secret_to_save: str) -> None:
    """
    Saves the given credentials in the config file at CONFIG_FILE_LOCATION, in the correct format for loading the
    credentials. Will create the directories and file if they do not already exist. Will completely overwrite the config
    file.

    :param client_id_to_save: The client id to be saved into the config file
    :param client_secret_to_save: The client secret to be saved into the config file
    :return: None
    """
    config_directory = "".join(CONFIG_FILE_LOCATION.split("/")[:-1])
    if not os.path.isdir(config_directory):
        os.makedirs(config_directory)

    config_file = open(CONFIG_FILE_LOCATION, 'w')

    config_file.write("Client ID: " + client_id_to_save + "\nClient Secret: " + client_secret_to_save)

    print("Credentials Saved in: " + CONFIG_FILE_LOCATION)


def get_client_context(client_id_in: str, client_secret_in: str) -> ClientContext:
    client_credentials = ClientCredential(client_id_in, client_secret_in)
    return ClientContext(SHAREPOINT_URL).with_credentials(client_credentials)


def get_id_to_datasets_map(ctx_in: ClientContext) -> dict[int, list[sm.Dataset]]:
    list_object = ctx_in.web.lists.get_by_title("Datasets")
    items = list_object.get_items()
    ctx_in.load(items)
    ctx_in.execute_query()

    id_to_datasets_map: dict[int, list[sm.Dataset]] = {}
    for item in items:
        record = item.properties

        dataset = sm.Dataset()

        project_id = record["ProjectID"]

        dataset.device_name = record["DeviceName"]
        dataset.device_type = record["DeviceName"]
        dataset.data_type = record["DatasetType"]
        dataset.top_left_latitude = record["TopLeftLatitude"]
        dataset.top_left_longitude = record["TopLeftLongitude"]
        dataset.bottom_right_latitude = record["BottomRightLatitude"]
        dataset.bottom_right_longitude = record["BottomRightLongitude"]
        dataset.data_size = record["DataSize"]

        if project_id not in id_to_datasets_map.keys():
            id_to_datasets_map[project_id] = []

        id_to_datasets_map[project_id].append(dataset)

    return id_to_datasets_map

def get_id_to_entities_map(ctx_in: ClientContext) -> dict[int, list[sm.Entity]]:
    list_object = ctx_in.web.lists.get_by_title("Project Entities")
    items = list_object.get_items()
    ctx_in.load(items)
    ctx_in.execute_query()

    id_to_entities_map: dict[int, list[sm.Entity]] = {}
    for item in items:
        record = item.properties

        entity = sm.Entity()

        project_id = record["ProjectID"]

        entity.name = record["Name"]
        entity.link = record["Url"]
        entity.type = record["EntityType"]

        if project_id not in id_to_entities_map.keys():
            id_to_entities_map[project_id] = []

        id_to_entities_map[project_id].append(entity)

    return id_to_entities_map


def get_projects(ctx_in: ClientContext) -> list[sm.Project]:
    id_to_datasets: list[sm.Dataset] = get_id_to_datasets_map(ctx_in)
    id_to_entities: list[sm.Entity] = get_id_to_entities_map(ctx_in)

    list_object = ctx_in.web.lists.get_by_title("Projects")
    items = list_object.get_items()
    ctx_in.load(items)
    ctx_in.execute_query()

    projects_out: list[sm.Project] = []
    for item in items:
        record = item.properties

        if record["Status"] == READY_STATUS:
            project  = sm.Project()

            id = record["Id"]

            project.name = record["Title"]
            project.country = record["Country"]
            project.latitude = record["Latitude"]
            project.longitude = record["Longitude"]
            project.project_description = record["ProjectDescription"]
            project.site_description = record["SiteDescription"]
            project.collection_date_start = record["CollectionStartDate"]
            project.collection_date_end = record["CollectionEndDate"]
            project.license_type = record["LicenseType"]
            project.license_link = sm.Project.get_license_link(project.license_type)
            project.external_project_link = record["ExternalProjectLink"]
            project.additional_info_link = record["AdditionalInfoLink"]
            project.keywords = record["Keywords"]

            project.datasets = id_to_datasets[id]
            project.entities = id_to_entities[id]

            projects_out.append(project)

    return projects_out


def get_survey_template_data() -> list[list[str]]:
    """
    Reads and parses the data from the survey template csv file to be used as a template.

    :return: A list of list of strings representing the survey template data
    """
    survey_template = open(SURVEY_TEMPLATE_PATH, 'r', encoding='Latin1')
    survey_template_reader = csv.reader(survey_template)

    survey_data_out = []
    for row in survey_template_reader:
        survey_data_out.append(row)

    survey_template.close()

    return survey_data_out


def add_dataset_data(survey_data_in: list[list[str]],
                     device_section_template_in: list[list[str]],
                     data_set_section_template_in: list[list[str]],
                     dataset_in: sm.Dataset,
                     dataset_index_in: int) -> list[list[str]]:
    device_section_data = deepcopy(device_section_template_in)
    device_section_data[sm.DEVICE_NAME_ROW_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.device_name
    device_section_data[sm.DEVICE_TYPE_ROW_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.device_type
    device_section_data[sm.DEVICE_DATA_TYPE_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.data_type

    data_set_section_data = deepcopy(data_set_section_template_in)
    data_set_section_data[sm.DATA_SET_TYPE_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.data_type
    data_set_section_data[sm.LATITUDE_TOP_LEFT_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.top_left_latitude
    data_set_section_data[sm.LONGITUDE_TOP_LEFT_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.top_left_longitude
    data_set_section_data[sm.LATITUDE_BOTTOM_RIGHT_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.bottom_right_latitude
    data_set_section_data[sm.LONGITUDE_BOTTOM_RIGHT_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.bottom_right_longitude
    data_set_section_data[sm.SIZE_OFFSET][sm.FILL_IN_COLUMN] = dataset_in.data_size

    device_section_insert_index = sm.ORIGINAL_DEVICE_SECTION + \
                                  sm.DEVICE_SECTION_LENGTH * dataset_index_in
    survey_data_in = survey_data_in[:device_section_insert_index] + \
                     device_section_data + \
                     survey_data_in[device_section_insert_index:]

    data_set_section_insert_index = sm.ORIGINAL_DEVICE_SECTION + \
                                    sm.DEVICE_SECTION_LENGTH * (dataset_index_in + 1) + \
                                    sm.DATA_SET_SECTION_LENGTH * dataset_index_in + 1
    survey_data_in = survey_data_in[:data_set_section_insert_index] + \
                     data_set_section_data + \
                     survey_data_in[data_set_section_insert_index:]

    return survey_data_in


def add_entity_data(survey_data_in: list[list[str]],
                    organization_section_template_in: list[list[str]],
                    entity_in: sm.Entity,
                    num_datasets_in: int,
                    entity_index_in: int) -> list[list[str]]:
    organization_data = deepcopy(organization_section_template_in)


    organization_data[sm.ORGANIZATION_NAME_OFFSET][sm.FILL_IN_COLUMN] = entity_in.name
    organization_data[sm.ORGANIZATION_URL_OFFSET][sm.FILL_IN_COLUMN] = entity_in.link
    organization_data[sm.ENTITY_TYPE_OFFSET][sm.FILL_IN_COLUMN] = entity_in.type

    organization_data_insert_index = sm.ORIGINAL_DEVICE_SECTION + \
                                     num_datasets_in * (sm.DEVICE_SECTION_LENGTH +
                                                            sm.DATA_SET_SECTION_LENGTH) + \
                                     entity_index_in * sm.ORGANIZATION_SECTION_LENGTH + \
                                     2

    survey_data = survey_data_in[:organization_data_insert_index] + \
                  organization_data + \
                  survey_data_in[organization_data_insert_index:]

    return survey_data


def fill_in_data(template_data: list[list[str]], project_in: sm.Project) -> list[list[str]]:
    survey_data = template_data

    # Fill in project data
    survey_data[sm.PROJECT_NAME][sm.FILL_IN_COLUMN] = project_in.name
    survey_data[sm.COUNTRY][sm.FILL_IN_COLUMN] = project_in.country
    survey_data[sm.LATITUDE][sm.FILL_IN_COLUMN] = project_in.latitude
    survey_data[sm.LONGITUDE][sm.FILL_IN_COLUMN] = project_in.longitude
    survey_data[sm.STATUS][sm.FILL_IN_COLUMN] = "Upcoming"
    survey_data[sm.PROJECT_DESCRIPTION][sm.FILL_IN_COLUMN] = project_in.project_description[3:-3]
    survey_data[sm.SITE_DESCRIPTION][sm.FILL_IN_COLUMN] = project_in.site_description[3:-3]
    survey_data[sm.START_DATE_ROW][sm.FILL_IN_COLUMN] = project_in.collection_date_start
    survey_data[sm.END_DATE_ROW][sm.FILL_IN_COLUMN] = project_in.collection_date_end

    (year, day, month) = str(datetime.date.today()).split("-")
    publish_date = year + "-" + day + "-" + month
    survey_data[sm.PUBLISH_DATE][sm.FILL_IN_COLUMN] = publish_date
    survey_data[sm.LICENSE_TYPE][sm.FILL_IN_COLUMN] = project_in.license_type
    survey_data[sm.LICENSE_LINK][sm.FILL_IN_COLUMN] = sm.Project.get_license_link(project_in.license_type)
    survey_data[sm.EXTERNAL_PROJECT_LINK][sm.FILL_IN_COLUMN] = project_in.external_project_link
    survey_data[sm.ADDITIONAL_INFO_LINK][sm.FILL_IN_COLUMN] = project_in.additional_info_link
    survey_data[sm.KEYWORDS][sm.FILL_IN_COLUMN] = project_in.keywords

    # Get template for specific sections
    device_section_template = survey_data[sm.ORIGINAL_DEVICE_SECTION:
                                          sm.ORIGINAL_DEVICE_SECTION + sm.DEVICE_SECTION_LENGTH]
    data_set_section_template = survey_data[sm.ORIGINAL_DATA_SET_SECTION:
                                            sm.ORIGINAL_DATA_SET_SECTION + sm.DATA_SET_SECTION_LENGTH]
    entity_section_template = survey_data[sm.ORIGINAL_ORGANIZATIONS_SECTION:
                                                sm.ORIGINAL_ORGANIZATIONS_SECTION + sm.ORGANIZATION_SECTION_LENGTH]

    # remove the data set sections for now
    survey_data = survey_data[:sm.ORIGINAL_ORGANIZATIONS_SECTION] + \
                  survey_data[sm.ORIGINAL_ORGANIZATIONS_SECTION + sm.ORGANIZATION_SECTION_LENGTH:]
    survey_data = survey_data[:sm.ORIGINAL_DATA_SET_SECTION] + \
                  survey_data[sm.ORIGINAL_DATA_SET_SECTION + sm.DATA_SET_SECTION_LENGTH:]
    survey_data = survey_data[:sm.ORIGINAL_DEVICE_SECTION] + \
                  survey_data[sm.ORIGINAL_DEVICE_SECTION + sm.DEVICE_SECTION_LENGTH:]

    # Fill out Device and Dataset Section
    for device_index in range(len(project_in.datasets)):
        dataset = project_in.datasets[device_index]

        survey_data = add_dataset_data(survey_data, device_section_template,
                                       data_set_section_template, dataset,
                                       device_index)

    # Fill out Entity Section
    for entity_index in range(len(project_in.entities)):
        entity = project_in.entities[entity_index]

        survey_data = add_entity_data(survey_data, entity_section_template,
                                      entity, len(project_in.datasets),
                                      entity_index)

    return survey_data


def create_filled_survey(filled_survey_data_in: list[list[str]], filled_survey_path_in: str) -> None:
    """
    Fills in some of the given survey data based on the given survey path.

    :param filled_survey_data_in: The list of list of strings representing the survey data to write
    :param filled_survey_path_in: The string path of the file to write to
    :return: None
    """
    filled_survey = open(filled_survey_path_in, 'w', encoding='utf-8', newline="")
    writer = csv.writer(filled_survey)

    for csv_row in filled_survey_data_in:
        writer.writerow(csv_row)

    filled_survey.close()


def process_projects(projects_in: list[sm.Project],
                     new_survey_directory_path_in: str) -> None:
    for current_project in projects_in:

        template_data = get_survey_template_data()
        filled_survey_data = fill_in_data(template_data, current_project)

        file_path = new_survey_directory_path_in + "/" + \
                    current_project.name + "_Survey.csv"

        file_path_index = 1
        while os.path.isfile(file_path):
            file_path = new_survey_directory_path_in + "/" + \
                        current_project.name + "_" + str(file_path_index) +\
                        "_Survey.csv"
            file_path_index += 1

        create_filled_survey(filled_survey_data, file_path)

        print("Project \"" + current_project.name + "\" downloaded to \"" + file_path + "\"")


if __name__== "__main__":
    if len(sys.argv) != 2:
        print("Please provide a directory to generate csv files.")
        exit()

    directory_path = sys.argv[1]
    if not os.path.isdir(directory_path):
        print("The provided directory does not exists.")
        exit()

    client_id, client_secret = get_user_credentials()

    ctx = get_client_context(client_id, client_secret)

    projects = get_projects(ctx)

    if len(projects) == 0:
        print("There are no project records with the \"Ready to be Published\" status.")
        exit()

    process_projects(projects, directory_path)
