# python db_tool_final.py Data/0xe0-7v64.csv
# python db_tool_final.py Data/Calidos_xhj8-8x48.csv
import os.path
import mysql.connector
import csv
import sys
import datetime

CONFIG_FILE_LOCATION = "config_files/mysql_uploader.config"
HOST = "208.109.25.83"
DATABASE = "OpenHeritage"


DOI_ROW_INDEX = 2
PROJECT_NAME_ROW_INDEX = 3
COUNTRY_ROW_INDEX = 4
LATITUDE_ROW_INDEX = 5
LONGITUDE_ROW_INDEX = 6
STATUS_ROW_INDEX = 7
PROJECT_DESCRIPTION_ROW_INDEX = 8
SITE_DESCRIPTION_ROW_INDEX = 9
COLLECTION_START_DATE_ROW_INDEX = 10
COLLECTION_END_DATE_ROW_INDEX = 11
PUBLISH_DATE_ROW_INDEX = 12
LICENSE_TYPE_ROW_INDEX = 13
LICENSE_LINK_ROW_INDEX = 14
EXTERNAL_PROJECT_LINK_INDEX = 15
ADDITIONAL_INFO_LINK_ROW_INDEX = 16
KEYWORDS_ROW_INDEX = 17

def get_project_master_record(values_in: list[str]) \
    -> tuple[str, str, str, str, str, str, str, str, str, str, str, str, str, str, str, str]:
    """
    Parses the given values and returns a tuple containing the relevant information for inserting into the
    Project_MASTER table. Reverses the collection_data_start, collection_date_end, and publish_date, from the
    day-month-year format to the year-month-day format. The license_link value is currently always set to
    "https://creativecommons.org/licenses/by/4.0" regardless of the given value. Any other value is taken from the given
    list of values.

    :param values_in: A list of strings, which contains the values of the doi project
    :return: A tuple representing the values of a record to be inserting into the Project_MASTER table. The tuple is in
             the following format: (doi, project_name, country, latitude, longitude, status, project_description,
                                    site_description, collection_date_start, collection_date_end, publish_date,
                                    license_type, license_link, external_project_link, additional_info_link, keywords)
    """
    doi = values_in[DOI_ROW_INDEX]
    project_name = values_in[PROJECT_NAME_ROW_INDEX]
    country = values_in[COUNTRY_ROW_INDEX]
    latitude = values_in[LATITUDE_ROW_INDEX]
    longitude = values_in[LONGITUDE_ROW_INDEX]
    # status = 'Published'
    # status = 'Upcoming'
    status = values_in[STATUS_ROW_INDEX]
    project_description = values_in[PROJECT_DESCRIPTION_ROW_INDEX]
    site_description = values_in[SITE_DESCRIPTION_ROW_INDEX]

    
    collection_start_date = values_in[COLLECTION_START_DATE_ROW_INDEX]
    if "/" in values_in[COLLECTION_START_DATE_ROW_INDEX]:
        [month, day, year] = values_in[COLLECTION_START_DATE_ROW_INDEX].split("/")
        collection_start_date = year + "-" + month + "-" + day

    collection_end_date = values_in[COLLECTION_END_DATE_ROW_INDEX]
    if "/" in values_in[COLLECTION_END_DATE_ROW_INDEX]:
        [month, day, year] = values_in[COLLECTION_END_DATE_ROW_INDEX].split("/")
        collection_end_date = year + "-" + month + "-" + day
    
    if values_in[PUBLISH_DATE_ROW_INDEX] == "":
        (year, day, month) = str(datetime.date.today()).split("-")
        publish_date = year + "-" + month + "-" + day
    elif "/" in values_in[PUBLISH_DATE_ROW_INDEX]:
        [month, day, year] = values_in[PUBLISH_DATE_ROW_INDEX].split("/")
        publish_date = year + "-" + month + "-" + day
    else:
        publish_date = values_in[PUBLISH_DATE_ROW_INDEX]

    license_type = values_in[LICENSE_TYPE_ROW_INDEX]
    license_link = values_in[LICENSE_LINK_ROW_INDEX]

    external_project_link = values_in[EXTERNAL_PROJECT_LINK_INDEX]
    additional_info_link = values_in[ADDITIONAL_INFO_LINK_ROW_INDEX]
    keywords = values_in[KEYWORDS_ROW_INDEX]

    # project master entry (oxford data did not places the link fixed here for all)
    return (doi, project_name, country, latitude, longitude, status, project_description, site_description,
            collection_start_date, collection_end_date, publish_date, license_type, license_link, external_project_link,
            additional_info_link, keywords)


DEVICE_NAME_INDEX_OFFSET = 0
DEVICE_TYPE_INDEX_OFFSET = 1
DEVICE_DATA_TYPE_INDEX_OFFSET = 2


def get_devices(names_in: list[str], values_in: list[str]) -> list[tuple[str, str, str]]:
    """
    Parses the given list of names and values and returns a list of tuples, which represent device information. It will
    not include any devices where all the information is empty.

    :param names_in: A list of names, which represent the names of the respective values in the values list.
    :param values_in: A list of strings, which contains the values of the doi project
    :return: A list of tuples, which hold data for devices used in the doi project. Each tuple is in the format:
             (device_name, device_type, data_type)
    """
    devices = []

    index = 0
    for name in names_in:
        if name == 'deviceName':
            device_name = values_in[index + DEVICE_NAME_INDEX_OFFSET]
            device_type = values_in[index + DEVICE_TYPE_INDEX_OFFSET]
            data_type = values_in[index + DEVICE_DATA_TYPE_INDEX_OFFSET]

            if device_name != "" or device_type != "" or data_type != "":
                device = (device_name, device_type, data_type)
                devices.append(device)

        index += 1

    return devices


DATA_TYPE_INDEX_OFFSET = -1
LATITUDE_TOP_LEFT_INDEX_OFFSET = 0
LONGITUDE_TOP_LEFT_INDEX_OFFSET = 1
LATITUDE_BOTTOM_RIGHT_INDEX_OFFSET = 2
LONGITUDE_BOTTOM_RIGHT_INDEX_OFFSET = 3
DATA_SIZE_INDEX_OFFSET = 4


def get_data_set_records(names_in: list[str], values_in: list[str], doi_in: str, project_name_in: str) \
    -> list[tuple[str, str, str, str, str, str, str, str, str]]:
    """
    Parses the given list of names and values and returns a list of tuples, which represent a record of a data set to be
    inserted into the Datasets table. Will not include any records where all of the dataset values are blank.

    :param names_in: A list of names, which represent the names of the respective values in the values list.
    :param values_in: A list of strings, which contains the values of the doi project
    :param doi_in: The doi of the project
    :param project_name_in: The name of the project
    :return: A list of tuples, which represent records to be inserted into the Datasets table. Each tuple is in the
             following format: (ID, DOI_Project, project_name, dataType, latitude_top_left, longitude_top_left,
                                latitude_bottom_right, longitude_bottom_right, dataSize)
    """
    data_sets = []

    index = 0
    for name in names_in:
        if name == 'latitude_top_left' and values_in[index + DATA_TYPE_INDEX_OFFSET] != '':
            data_type = values_in[index + DATA_TYPE_INDEX_OFFSET]

            latitude_top_left = values_in[index + LATITUDE_TOP_LEFT_INDEX_OFFSET]
            longitude_top_left = values_in[index + LONGITUDE_TOP_LEFT_INDEX_OFFSET]
            latitude_bottom_right = values_in[index + LATITUDE_BOTTOM_RIGHT_INDEX_OFFSET]
            longitude_bottom_right = values_in[index + LONGITUDE_BOTTOM_RIGHT_INDEX_OFFSET]

            data_size = values_in[index + DATA_SIZE_INDEX_OFFSET]

            if data_type != "" or latitude_top_left != "" or longitude_top_left != "" or \
               latitude_bottom_right != "" or longitude_bottom_right != "" or data_size != "":
                data_set = ("", doi_in, project_name_in, data_type, latitude_top_left, longitude_top_left,
                            latitude_bottom_right, longitude_bottom_right, data_size)
                data_sets.append(data_set)

        index += 1
        
    return data_sets


DATA_SET_ID_INDEX = 0


def get_data_device_records(data_sets_in: list[tuple[str, str, str, str, str, str, str, str, str]],
                            device_ids_in: list[str]) -> list[tuple[str, str]]:
    """
    Parses the data_sets and matches the dataset id with the id of the device used to create the dataset and returns a
    list of tuples, which contain the dataset id and device id.

    :param data_sets_in: A list of tuples containing data on each of the datasets in the project. Each tuple is in the
           format: (ID, DOI_Project, project_name, dataType, latitude_top_left, longitude_top_left,
                    latitude_bottom_right, longitude_bottom_right, dataSize)
    :param device_ids_in: A list of device ids, in the same order as their respective dataset in data_sets_in
    :return: A list of tuples, which are in the following format: (id_Dataset, id_Device)
    """
    data_device_records = []

    index = 0
    for data_set in data_sets_in:

        data_device_record = (data_set[DATA_SET_ID_INDEX], device_ids_in[index])
        data_device_records.append(data_device_record)

        index += 1

    return data_device_records


ORGANIZATION_NAME_INDEX_OFFSET = 0
ORGANIZATION_URL_INDEX_OFFSET = 1
ENTITY_TYPE_INDEX_OFFSET = 2


def get_organizations(names_in: list[str], values_in: list[str]) -> list[tuple[str, str, str]]:
    """
    Parses the given list of names and values and returns a list of tuples, which represent an organization that has
    contributed to the project.

    :param names_in: A list of names, which represent the names of the respective values in the values list.
    :param values_in: A list of strings, which contains the values of the doi project
    :return: A list of tuples, which contain data for an organization the contributed to the project. Each tuple is in
             the following format: (organization_name, organization_url, entity_type)
    """
    organizations = []

    index = 0
    for name in names_in:
        if name == 'organizationName':
            organization_name = values_in[index + ORGANIZATION_NAME_INDEX_OFFSET]
            organization_url = values_in[index + ORGANIZATION_URL_INDEX_OFFSET]
            entity_type = values_in[index + ENTITY_TYPE_INDEX_OFFSET]

            if organization_name != "":
                organization = (organization_name, organization_url, entity_type)
                organizations.append(organization)

        index += 1

    return organizations


def print_record_changes(project_master_record_in: tuple[str, str, str, str, str, str, str, str, str, str, str, str,
                                                         str, str, str, str],
                         project_master_update_record_in: tuple[str, str, str, str, str, str, str, str, str, str, str,
                                                                 str, str, str, str, str, str],
                         device_records_in: list[tuple[str, str]],
                         data_set_records_in: list[tuple[str, str, str, str, str, str, str, str, str]],
                         data_device_records_in: list[tuple[str, str]],
                         organization_records_in: list[tuple[str, str]]) -> None:
    """
    Prints the results of the record changes from the given lists of records. If a table was not updated or changed in
    any way then t

    :param project_master_record_in: The record that may have been inserted into the Project_MASTER table. If
                                     project_master_update_record_in is empty, then this record was inserted instead.
    :param project_master_update_record_in: The record that may have been used to update a record in the Project_MASTER
                                            table. If this value is empty, then project_master_record_in was inserted.
    :param device_records_in: The list of device related records that was inserted into the Devices table
    :param data_set_records_in: The list of dataset related records that was inserted into the datasets table
    :param data_device_records_in: The list of data device records that was inserted into the Data_Device table
    :param organization_records_in: The list of organization related records that was inserted into Organizations table
    :return: None
    """
    print("The following records were inserted into the database")
    print("##########################################################################")
    if project_master_update_record_in == "":
        print("Successfully inserted the following record into the Projects_MASTER table:")
        print(project_master_record_in)
        print("##########################################################################")
    else:
        print("Successfully updated the following record into the Projects_MASTER table:")
        print(project_master_update_record_in)
        print("##########################################################################")

    if len(device_records_in) > 0:
        print("Successfully inserted the following records into the Devices table:")
        for device_record in device_records_in:
            print(device_record)
        print("##########################################################################")

    if len(data_set_records_in) > 0:
        print("Successfully inserted the following records into the Datasets table:")
        for data_set_record in data_set_records_in:
            print(data_set_record)
        print("##########################################################################")

    if len(data_device_records_in) > 0:
        print("Successfully inserted the following records into the Data_Device table:")
        for data_device_record in data_device_records_in:
            print(data_device_record)
        print("##########################################################################")

    if len(organization_records_in) > 0:
        print("Successfully inserted the following records into the Organizations table:")
        for organization_record in organization_records_in:
            print(organization_record)
        print("##########################################################################")


def main(file_path: str) -> None:
    """
    Uploads the data in the csv file path to the OpenHeritage HeidiSQL database. Connects to host='208.109.25.83',
    database='OpenHeritage'. Credentials will be automatically filled in from config_file/mysql_uploader.config if it
    exists. If it does not exist of has invalid credentials, then the user will be prompted to enter a username and
    password for the database.

    :param file_path: The file path to the csv to be uploaded.
    :return: None
    """
    # read csv file
    f = open(file_path, 'r')

    # parse csv file
    with f:
        reader = csv.reader(f)

        names = []
        values = []

        for row in reader:
            values.append(row[3])
            names.append(row[0])

        ###Insert Record to Projects_MASTER###
        project_master_insert_record = get_project_master_record(values)
        (doi, project_name, null, null, null, null, null, null,
            null, null, null, null, null,
            null, null, null) = project_master_insert_record
        devices = get_devices(names, values)
        organizations = get_organizations(names, values)
        data_set_records = get_data_set_records(names, values, doi, project_name)

        upload_record(project_master_insert_record, devices, organizations, data_set_records)

def upload_record(project_master_insert_record_in: tuple[str, str, str, str, str, str, str, str, str, str, str, str, str, str, str, str],
                  devices_in: list[tuple[str, str, str]], organizations_in: list[tuple[str, str, str]],
                  data_set_records_in: list[tuple[str, str, str, str, str, str, str, str, str]]) -> None:
    username, password = get_user_credentials()
    
    #Connect to Mysql database
    #cyark host 23.229.231.70
    try:
        connection = mysql.connector.connect(host=HOST,
                                             database=DATABASE,
                                             user=username,
                                             password=password)
        if connection.is_connected():
            db_info = connection.get_server_info()
            print("Connected to MySQL Server version ", db_info)
        else:
            print("Error: Failed to Connect to MYSQL Server")
            exit()

        (doi, project_name, country, latitude, longitude, status, project_description, site_description,
            collection_start_date, collection_end_date, publish_date, license_type, license_link,
            external_project_link, additional_info_link, keywords) = project_master_insert_record_in

        try:
            # Check if doi in Projects_MASTER already
            search_query = "SELECT * FROM Projects_MASTER WHERE doi = %s"

            cursor = connection.cursor()
            cursor.execute(search_query, (doi,))

            query_result = cursor.fetchall()

            project_master_update_record = ""
            if query_result:
                mysql_update_query_project_master = """UPDATE Projects_MASTER SET doi = %s, project_name = %s, country =
                                                        %s, latitude = %s,
                                                        longitude = %s, status = %s, project_description = %s, site_description = %s,
                                                        collection_date_start = %s, collection_date_end = %s, publish_date = %s,
                                                        license_type = %s, license_link = %s, external_project_link = %s,
                                                        additional_info_link = %s, keywords = %s WHERE doi = %s"""
                project_master_update_record = (doi, project_name, country, latitude, longitude, status,
                                                project_description, site_description, collection_start_date,
                                                collection_end_date, publish_date, license_type, license_link,
                                                external_project_link, additional_info_link, keywords, doi)
                cursor.execute(mysql_update_query_project_master, project_master_update_record)
            else:
                mysql_insert_query_project_master = """INSERT INTO Projects_MASTER (doi, project_name, country, 
                                                        latitude, longitude, status, project_description, 
                                                        site_description, collection_date_start, collection_date_end, 
                                                        publish_date, license_type, license_link, external_project_link,
                                                        additional_info_link, keywords) VALUES (%s, %s, %s, %s, %s, %s,
                                                        %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)"""

                cursor = connection.cursor()
                cursor.execute(mysql_insert_query_project_master, project_master_insert_record_in)

            ###Insert Device into Devices###
            search_query = "SELECT * FROM Devices WHERE deviceName = %s"
            mysql_insert_query_devices = "INSERT INTO Devices (deviceName, deviceType) VALUES (%s,%s)"
            
            device_ids = []
            device_names = []
            device_data_types = []

            device_records = []
            for device in devices_in:
                if device[0] != '':
                    # search device name in database
                    cursor.execute(search_query, (device[0],))
                    my_device_result = cursor.fetchall()

                    if my_device_result:
                        device_name = my_device_result[0][0]
                        device_type = my_device_result[0][1]
                        data_type = device[2]

                        if device_type == "" and device[1] != "":
                            mysql_update_query_devices = "UPDATE Devices SET deviceType = %s WHERE deviceName = %s"

                            cursor.execute(mysql_update_query_devices, (device[1], device_name))

                        device_ids.append(device_name)
                        device_names.append(device_type)
                        device_data_types.append(data_type)
                    else:
                        devices_record = (device[0], device[1])
                        device_records.append(devices_record)
                        cursor.execute(mysql_insert_query_devices, devices_record)

                        # search device name in database
                        search_tuple = (device[0],)
                        cursor.execute(search_query, search_tuple)
                        my_device_result = cursor.fetchall()

                        device_ids.append(my_device_result[0][0])
                        device_names.append(my_device_result[0][1])
                        device_data_types.append(device[2])

            ###Query Data Device records for this doi###
            mysql_query_datasets = "SELECT * FROM Datasets WHERE DOI_Project = %s"
            cursor.execute(mysql_query_datasets, (doi, ))
            database_doi_datasets = cursor.fetchall()

            database_doi_dataset_ids = []
            for database_doi_dataset in database_doi_datasets:
                database_doi_dataset_id = database_doi_dataset[0]
                database_doi_dataset_ids.append(database_doi_dataset_id)

            mysql_delete_data_devices = "DELETE FROM Data_Device WHERE id_Dataset = %s"
            for database_doi_dataset_id in database_doi_dataset_ids:
                cursor.execute(mysql_delete_data_devices, (database_doi_dataset_id,))

            ###Delete Dataset records for this doi###
            mysql_delete_datasets = "DELETE FROM Datasets WHERE DOI_Project = %s"

            cursor.execute(mysql_delete_datasets, (doi,))
            ###Insert Record into Datasets###
            mysql_insert_query_datasets = """INSERT INTO Datasets (ID, DOI_Project, project_name, dataType, 
                                                latitude_top_left, longitude_top_left, latitude_bottom_right, 
                                                longitude_bottom_right, dataSize) VALUES (%s, %s, %s, %s, %s, %s, %s, 
                                                %s, %s)"""
            

            updated_data_set_records = []
            for data_set_record in data_set_records_in:
                cursor.execute(mysql_insert_query_datasets, data_set_record)

                # get the generated id from the dataset table
                data_set_record = (cursor.lastrowid, data_set_record[1], data_set_record[2], data_set_record[3],
                                    data_set_record[4], data_set_record[5], data_set_record[6], data_set_record[7],
                                    data_set_record[8])
                updated_data_set_records.append(data_set_record)
            data_set_records = updated_data_set_records

            ###Insert Record into Data_Devices###
            mysql_insert_query_data_devices = """INSERT INTO Data_Device (id_Dataset, id_Device)
                                                    VALUES (%s,%s) """

            data_device_records = get_data_device_records(data_set_records, device_ids)

            for data_device_record in data_device_records:
                cursor.execute(mysql_insert_query_data_devices, data_device_record)

            ###Insert Record into Organizations###
            search_organization_query = "SELECT * FROM Organizations WHERE organizationName = %s"
            mysql_insert_query_organizations = """INSERT IGNORE INTO Organizations (organizationName, organizationURL)
                                                    VALUES (%s,%s)"""
            

            organization_ids = []
            organization_names = []
            entity_type_list = []

            organization_records = []
            for organization in organizations_in:
                # check if the csv field is empty
                if organization[0] != '':
                    # search tuple to search the name of the organization (every third element in array)
                    search_organization_tuple = (organization[0],)
                    cursor.execute(search_organization_query, search_organization_tuple)

                    organization_result = cursor.fetchall()

                    if not organization_result:
                        # INSERT
                        organization_record = (organization[0], organization[1])
                        cursor.execute(mysql_insert_query_organizations, organization_record)
                        organization_records.append(organization_record)

                        # Search the insert again for ID
                        search_organization_tuple = (organization[0],)
                        cursor.execute(search_organization_query, search_organization_tuple)

                        organization_result = cursor.fetchall()

                    # Add the organization to the arrays
                    organization_ids.append(organization_result[0][0])
                    organization_names.append(organization_result[0][1])
                    entity_type_list.append(organization[2])

            ###Delete Project Entities records for this doi###
            mysql_delete_project_entities = "DELETE FROM Project_Entities WHERE DOI_Project = %s"

            cursor.execute(mysql_delete_project_entities, (doi,))

            ### Insert Project_Entities ###
            mysql_insert_query_project_entities = """INSERT INTO Project_Entities (DOI_Project, project_name, id_Organization, isAuthority, isCollector, isFunder, isPartner, isContributor)
                                                        VALUES (%s,%s, %s, %s, %s, %s, %s, %s)"""

            for index in range(len(organization_names)):
                if entity_type_list[index] == 'collector' or entity_type_list[index] == 'Collector':
                    project_entities_tuple = (doi, project_name, organization_ids[index], 0, 1, 0, 0, 0)
                    # print("Organizations inserted", organization_ids[index], organization_names[index])

                elif entity_type_list[index] == 'partner' or entity_type_list[index] == 'Partner':
                    project_entities_tuple = (doi, project_name, organization_ids[index], 0, 0, 0, 1, 0)
                    # print("Organizations inserted", organization_ids[index], organization_names[index])

                elif entity_type_list[index] == 'contributor' or entity_type_list[index] == 'Contributor':
                    project_entities_tuple = (doi, project_name, organization_ids[index], 0, 0, 0, 0, 1)
                    # print("Organizations inserted", organization_ids[index], organization_names[index])

                elif entity_type_list[index] == 'funder' or entity_type_list[index] == 'Funder':
                    project_entities_tuple = (doi, project_name, organization_ids[index], 0, 0, 1, 0, 0)
                    # print("Organizations inserted", organization_ids[index], organization_names[index])

                elif entity_type_list[index] == 'Site_Authority':
                    project_entities_tuple = (doi, project_name, organization_ids[index], 1, 0, 0, 0, 0)
                    # print("Organizations inserted", organization_ids[index], organization_names[index])
                else:
                    project_entities_tuple = (doi, project_name, organization_ids[index], 0, 0, 0, 0, 0)

                cursor.execute(mysql_insert_query_project_entities, project_entities_tuple)
            print_record_changes(project_master_insert_record_in, project_master_update_record, device_records,
                                    data_set_records, data_device_records, organization_records)

            connection.commit()

            cursor.close()


        except mysql.connector.Error as error:
            print("Connection error", error.errno, error.msg)

        # close the connection
        finally:
            if connection.is_connected():
                cursor.close()
                connection.close()
                print("MySQL connection is closed")
    except mysql.connector.Error as error:
        print("Connection error", error.errno, error.msg)

def get_user_credentials() -> tuple([str, str]):
    if os.path.isfile(CONFIG_FILE_LOCATION):
        username, password = read_config_file()

        if not is_valid_credentials(username, password):
            print("Config file credentials are invalid.")
            username, password = prompt_user_for_credentials()
    else:
        username, password = prompt_user_for_credentials()
        
    return username, password    


def read_config_file() -> tuple[str, str]:
    """
    Parses the config file at CONFIG_FILE_LOCATION and returns the found username and password in a tuple. Will return
    blank values for the username and password if it cannot find them. These credentials are not guaranteed to work.

    :return: A tuple, representing the credentials stored in the config file, in the following format: (config_username,
             config_password)
    """
    print("Reading OpenHeritage Database Credentials from " + CONFIG_FILE_LOCATION + ".")

    config_file = open(CONFIG_FILE_LOCATION, 'r')

    config_username = ""
    config_password = ""
    for line in config_file.readlines():
        line_words = line.split(": ")

        if line_words[0] == "Username":
            config_username = line_words[1]
        elif line_words[0] == "Password":
            config_password = line_words[1]

    return config_username, config_password


def prompt_user_for_credentials() -> tuple[str, str]:
    """
    Prompts the user to manually input a username and then password. Will test the credentials and prompt the user to
    try again if credentials are not correct. The returned credentials are guaranteed to be valid.

    :return: A tuple, representing credentials provided by the user, which are guaranteed to be correct
    """
    while True:
        username_input = input("Enter OpenHeritage Username: ")
        password_input = input("Enter OpenHeritage Password: ")

        if is_valid_credentials(username_input, password_input):
            save_credentials(username_input, password_input)

            return username_input, password_input

        print("Entered credentials are invalid. Please try again.")


def is_valid_credentials(username_in: str, password_in: str) -> bool:
    """
    Tests the given username and password and returns whether the credentials are valid by attempting to connect to the
    database with them.

    :param username_in: The username of the credentials to test
    :param password_in: The password of the credentials to test
    :return: Whether the credentials are valid
    """
    try:
        connection = mysql.connector.connect(host='208.109.25.83',
                                             database='OpenHeritage',
                                             user=username_in,
                                             password=password_in)

        if connection.is_connected():
            connection.close()
            return True
        else:
            return False

    except mysql.connector.Error:
        return False


def save_credentials(username_to_save: str, password_to_save: str) -> None:
    """
    Saves the given credentials in the config file at CONFIG_FILE_LOCATION, in the correct format for loading the
    credentials. Will create the directories and file if they do not already exist. Will completely overwrite the config
    file.

    :param username_to_save: The username to be saved into the config file
    :param password_to_save: The password to be saved into the config file
    :return: None
    """
    config_directory = "".join(CONFIG_FILE_LOCATION.split("/")[:-1])
    if not os.path.isdir(config_directory):
        os.makedirs(config_directory)

    config_file = open(CONFIG_FILE_LOCATION, 'w')

    config_file.write("Username: " + username_to_save + "\nPassword: " + password_to_save)

    print("Credentials Saved in: " + CONFIG_FILE_LOCATION)


if __name__== "__main__":
    if len(sys.argv) != 2:
        raise ValueError('Please provide a csv to insert to database.')

    main(sys.argv[1])