import pyodbc
import sys
import csv
from copy import deepcopy


import survey_metadata as sm

# Microsoft Access Database specific constants
# Project_MASTER index constants
PROJECT_ID = 0
PROJECT_NAME_INDEX = 1
COUNTRY_INDEX = 2
LATITUDE_INDEX = 3
LONGITUDE_INDEX = 4
PROJECT_DESCRIPTION_INDEX = 5
SITE_DESCRIPTION_INDEX = 6
COLLECTION_DATE_START_INDEX = 7
COLLECTION_DATE_END_INDEX = 8
LICENSE_TYPE_INDEX = 9
EXTERNAL_PROJECT_LINK_INDEX = 10
ADDITIONAL_INFO_LINK_INDEX = 11
KEYWORDS_INDEX = 12

LIDAR_TERRESTRIAL_DEVICE_NAME_INDEX = 13
LIDAR_TERRESTRIAL_DEVICE_TYPE_INDEX = 14
LIDAR_TERRESTRIAL_LATITUDE_TOP_LEFT_INDEX = 15
LIDAR_TERRESTRIAL_LONGITUDE_TOP_LEFT_INDEX = 16
LIDAR_TERRESTRIAL_LATITUDE_BOTTOM_RIGHT_INDEX = 17
LIDAR_TERRESTRIAL_LONGITUDE_BOTTOM_RIGHT_INDEX = 18
LIDAR_TERRESTRIAL_DATA_SIZE_INDEX = 19

LIDAR_AERIAL_DEVICE_NAME_INDEX = 20
LIDAR_AERIAL_DEVICE_TYPE_INDEX = 21
LIDAR_AERIAL_LATITUDE_TOP_LEFT_INDEX = 22
LIDAR_AERIAL_LONGITUDE_TOP_LEFT_INDEX = 23
LIDAR_AERIAL_LATITUDE_BOTTOM_RIGHT_INDEX = 24
LIDAR_AERIAL_LONGITUDE_BOTTOM_RIGHT_INDEX = 25
LIDAR_AERIAL_DATA_SIZE_INDEX = 26

PHOTOGRAMMETRY_TERRESTRIAL_DEVICE_NAME_INDEX = 27
PHOTOGRAMMETRY_TERRESTRIAL_DEVICE_TYPE_INDEX = 28
PHOTOGRAMMETRY_TERRESTRIAL_LATITUDE_TOP_LEFT_INDEX = 29
PHOTOGRAMMETRY_TERRESTRIAL_LONGITUDE_TOP_LEFT_INDEX = 30
PHOTOGRAMMETRY_TERRESTRIAL_LATITUDE_BOTTOM_RIGHT_INDEX = 31
PHOTOGRAMMETRY_TERRESTRIAL_LONGITUDE_BOTTOM_RIGHT_INDEX = 32
PHOTOGRAMMETRY_TERRESTRIAL_DATA_SIZE_INDEX = 33

PHOTOGRAMMETRY_AERIAL_DEVICE_NAME_INDEX = 34
PHOTOGRAMMETRY_AERIAL_DEVICE_TYPE_INDEX = 35
PHOTOGRAMMETRY_AERIAL_LATITUDE_TOP_LEFT_INDEX = 36
PHOTOGRAMMETRY_AERIAL_LONGITUDE_TOP_LEFT_INDEX = 37
PHOTOGRAMMETRY_AERIAL_LATITUDE_BOTTOM_RIGHT_INDEX = 38
PHOTOGRAMMETRY_AERIAL_LONGITUDE_BOTTOM_RIGHT_INDEX = 39
PHOTOGRAMMETRY_AERIAL_DATA_SIZE_INDEX = 40

DATA_DERIVATIVES_DSM_ORTHO_DEVICE_NAME_INDEX = 41
DATA_DERIVATIVES_DSM_ORTHO_DEVICE_TYPE_INDEX = 42
DATA_DERIVATIVES_DSM_ORTHO_LATITUDE_TOP_LEFT_INDEX = 43
DATA_DERIVATIVES_DSM_ORTHO_LONGITUDE_TOP_LEFT_INDEX = 44
DATA_DERIVATIVES_DSM_ORTHO_LATITUDE_BOTTOM_RIGHT_INDEX = 45
DATA_DERIVATIVES_DSM_ORTHO_LONGITUDE_BOTTOM_RIGHT_INDEX = 46
DATA_DERIVATIVES_DSM_ORTHO_DATA_SIZE_INDEX = 47

DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DEVICE_NAME_INDEX = 48
DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DEVICE_TYPE_INDEX = 49
DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LATITUDE_TOP_LEFT_INDEX = 50
DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LONGITUDE_TOP_LEFT_INDEX = 51
DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LATITUDE_BOTTOM_RIGHT_INDEX = 52
DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LONGITUDE_BOTTOM_RIGHT_INDEX = 53
DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DATA_SIZE_INDEX = 54

# Project_Entities index constants
PROJECT_ID_INDEX = 1
ORGANIZATION_NAME_INDEX = 2
ORGANIZATION_URL_INDEX = 3
ENTITY_TYPE_INDEX = 4

SURVEY_TEMPLATE_PATH = "CSV_files/OH_Metadata_Survey_Template.csv"

def make_cursor(microsoft_access_file_path_in: str) -> pyodbc.Cursor:
    conn = pyodbc.connect("Driver={Microsoft Access Driver (*.mdb, *.accdb)};" +
                          "DBQ=" + microsoft_access_file_path_in + ";')")
    return conn.cursor()

def parse_project_master(cursor_in: pyodbc.Cursor) -> list[pyodbc.Row]:
    project_master_data = []
    cursor_in.execute('select * from Projects_MASTER')
    for row in cursor_in.fetchall():
        project_master_data.append(row)

    return project_master_data

def parse_project_entities(cursor_in: pyodbc.Cursor) -> dict[str, list[pyodbc.Row]]:
    project_entities_data = {}
    cursor_in.execute('select * from Project_Entities')
    for row in cursor_in.fetchall():
        project_id = row[PROJECT_ID_INDEX]
        if project_id in project_entities_data:
            project_entities_data[project_id].append(row)
        else:
            project_entities_data[project_id] = [row]

    return project_entities_data

def process_projects(projects_in: list[pyodbc.Row], entities_in: dict[str, list[pyodbc.Row]],
                     new_survey_directory_path_in: str):
    for project in projects_in:
        survey = get_survey_data(project, entities_in)
        file_path = new_survey_directory_path_in + "/" + \
                    project[PROJECT_NAME_INDEX] + "_Survey.csv"

        create_filled_survey(survey, file_path)

def get_survey_data(project_in: pyodbc.Row, entities_in: dict[str, list[pyodbc.Row]])\
        -> list[list[str]]:
    survey_data = get_survey_template_data()

    survey_data[sm.PROJECT_NAME][sm.FILL_IN_COLUMN] = project_in[PROJECT_NAME_INDEX]
    survey_data[sm.COUNTRY][sm.FILL_IN_COLUMN] = project_in[COUNTRY_INDEX]
    survey_data[sm.LATITUDE][sm.FILL_IN_COLUMN] = project_in[LATITUDE_INDEX]
    survey_data[sm.LONGITUDE][sm.FILL_IN_COLUMN] = project_in[LONGITUDE_INDEX]
    survey_data[sm.STATUS][sm.FILL_IN_COLUMN] = "Upcoming"
    survey_data[sm.PROJECT_DESCRIPTION][sm.FILL_IN_COLUMN] = project_in[PROJECT_DESCRIPTION_INDEX]
    survey_data[sm.SITE_DESCRIPTION][sm.FILL_IN_COLUMN] = project_in[SITE_DESCRIPTION_INDEX]

    collection_start_date = str(project_in[COLLECTION_DATE_START_INDEX])[:-9]
    [year, month, day] = collection_start_date.split("-")
    collection_start_date = day + "-" + month + "-" + year

    collection_end_date = str(project_in[COLLECTION_DATE_END_INDEX])[:-9]
    [year, month, day] = collection_end_date.split("-")
    collection_end_date = day + "-" + month + "-" + year

    print(collection_end_date)
    survey_data[sm.START_DATE_ROW][sm.FILL_IN_COLUMN] = collection_start_date
    survey_data[sm.END_DATE_ROW][sm.FILL_IN_COLUMN] = collection_end_date
    survey_data[sm.PUBLISH_DATE][sm.FILL_IN_COLUMN] = ""

    license_type = project_in[LICENSE_TYPE_INDEX]
    survey_data[sm.LICENSE_TYPE][sm.FILL_IN_COLUMN] = license_type

    if license_type == "CC BY":
        license_link = "https://creativecommons.org/licenses/by/4.0"
    elif license_type == "CC BY-SA":
        license_link = "https://creativecommons.org/licenses/by-sa/4.0"
    elif license_type == "CC BY-ND":
        license_link = "https://creativecommons.org/licenses/by-nd/4.0"
    elif license_type == "CC BY-NC":
        license_link = "https://creativecommons.org/licenses/by-nc/4.0"
    elif license_type == "CC BY-NC-SA":
        license_link = "https://creativecommons.org/licenses/by-nc-sa/4.0"
    elif license_type == "CC BY-NC-ND":
        license_link = "https://creativecommons.org/licenses/by-nc-nd/4.0"

    survey_data[sm.LICENSE_LINK][sm.FILL_IN_COLUMN] = license_link
    survey_data[sm.EXTERNAL_PROJECT_LINK][sm.FILL_IN_COLUMN] = project_in[EXTERNAL_PROJECT_LINK_INDEX]
    survey_data[sm.ADDITIONAL_INFO_LINK][sm.FILL_IN_COLUMN] = project_in[ADDITIONAL_INFO_LINK_INDEX]
    survey_data[sm.KEYWORDS][sm.FILL_IN_COLUMN] = project_in[KEYWORDS_INDEX]

    # Get template for specific sections
    device_section_template = survey_data[sm.ORIGINAL_DEVICE_SECTION :
                            sm.ORIGINAL_DEVICE_SECTION + sm.DEVICE_SECTION_LENGTH]
    data_set_section_template = survey_data[sm.ORIGINAL_DATA_SET_SECTION :
                        sm.ORIGINAL_DATA_SET_SECTION + sm.DATA_SET_SECTION_LENGTH]
    organization_section_template = survey_data[sm.ORIGINAL_ORGANIZATIONS_SECTION :
                        sm.ORIGINAL_ORGANIZATIONS_SECTION + sm.ORGANIZATION_SECTION_LENGTH]

    # remove the data set sections for now
    survey_data = survey_data[:sm.ORIGINAL_ORGANIZATIONS_SECTION] + \
                  survey_data[sm.ORIGINAL_ORGANIZATIONS_SECTION + sm.ORGANIZATION_SECTION_LENGTH:]
    survey_data = survey_data[:sm.ORIGINAL_DATA_SET_SECTION] + \
                  survey_data[sm.ORIGINAL_DATA_SET_SECTION + sm.DATA_SET_SECTION_LENGTH:]
    survey_data = survey_data[:sm.ORIGINAL_DEVICE_SECTION] + \
                  survey_data[sm.ORIGINAL_DEVICE_SECTION + sm.DEVICE_SECTION_LENGTH:]

    num_data_sets_added = 0

    # Data set section
    if project_in[LIDAR_TERRESTRIAL_DEVICE_NAME_INDEX] is not None and \
       project_in[LIDAR_TERRESTRIAL_DEVICE_NAME_INDEX] != "":
        device_name = project_in[LIDAR_TERRESTRIAL_DEVICE_NAME_INDEX]
        device_type = project_in[LIDAR_TERRESTRIAL_DEVICE_TYPE_INDEX]
        lat_top_left = project_in[LIDAR_TERRESTRIAL_LATITUDE_TOP_LEFT_INDEX]
        long_top_left = project_in[LIDAR_TERRESTRIAL_LONGITUDE_TOP_LEFT_INDEX]
        lat_bottom_right = project_in[LIDAR_TERRESTRIAL_LATITUDE_BOTTOM_RIGHT_INDEX]
        long_bottom_right = project_in[LIDAR_TERRESTRIAL_LONGITUDE_BOTTOM_RIGHT_INDEX]
        data_set_size = project_in[LIDAR_TERRESTRIAL_DATA_SIZE_INDEX]

        survey_data = add_data_set_data(survey_data, device_section_template,
                          data_set_section_template, device_name,
                          device_type, "LiDAR - Terrestrial", lat_top_left,
                          long_top_left, lat_bottom_right, long_bottom_right,
                          data_set_size, num_data_sets_added)
        num_data_sets_added += 1


    if project_in[LIDAR_AERIAL_DEVICE_NAME_INDEX] is not None and \
       project_in[LIDAR_AERIAL_DEVICE_NAME_INDEX] != "":
        device_name = project_in[LIDAR_AERIAL_DEVICE_NAME_INDEX]
        device_type = project_in[LIDAR_AERIAL_DEVICE_TYPE_INDEX]
        lat_top_left = project_in[LIDAR_AERIAL_LATITUDE_TOP_LEFT_INDEX]
        long_top_left = project_in[LIDAR_AERIAL_LONGITUDE_TOP_LEFT_INDEX]
        lat_bottom_right = project_in[LIDAR_AERIAL_LATITUDE_BOTTOM_RIGHT_INDEX]
        long_bottom_right = project_in[LIDAR_AERIAL_LONGITUDE_BOTTOM_RIGHT_INDEX]
        data_set_size = project_in[LIDAR_AERIAL_DATA_SIZE_INDEX]

        survey_data = add_data_set_data(survey_data, device_section_template,
                                        data_set_section_template, device_name,
                                        device_type, "LiDAR - Aerial", lat_top_left,
                                        long_top_left, lat_bottom_right, long_bottom_right,
                                        data_set_size, num_data_sets_added)
        num_data_sets_added += 1


    if project_in[PHOTOGRAMMETRY_TERRESTRIAL_DEVICE_NAME_INDEX] is not None and \
       project_in[PHOTOGRAMMETRY_TERRESTRIAL_DEVICE_NAME_INDEX] == "":
        device_name = project_in[PHOTOGRAMMETRY_TERRESTRIAL_DEVICE_NAME_INDEX]
        device_type = project_in[PHOTOGRAMMETRY_TERRESTRIAL_DEVICE_TYPE_INDEX]
        lat_top_left = project_in[PHOTOGRAMMETRY_TERRESTRIAL_LATITUDE_TOP_LEFT_INDEX]
        long_top_left = project_in[PHOTOGRAMMETRY_TERRESTRIAL_LONGITUDE_TOP_LEFT_INDEX]
        lat_bottom_right = project_in[PHOTOGRAMMETRY_TERRESTRIAL_LATITUDE_BOTTOM_RIGHT_INDEX]
        long_bottom_right = project_in[PHOTOGRAMMETRY_TERRESTRIAL_LONGITUDE_BOTTOM_RIGHT_INDEX]
        data_set_size = project_in[PHOTOGRAMMETRY_TERRESTRIAL_DATA_SIZE_INDEX]

        survey_data = add_data_set_data(survey_data, device_section_template,
                          data_set_section_template, device_name,
                          device_type, "Photogrammetry - Terrestrial",
                          lat_top_left, long_top_left, lat_bottom_right,
                          long_bottom_right, data_set_size, num_data_sets_added)
        num_data_sets_added += 1

    if project_in[PHOTOGRAMMETRY_AERIAL_DEVICE_NAME_INDEX] is not None and \
       project_in[PHOTOGRAMMETRY_AERIAL_DEVICE_NAME_INDEX] != "":
        device_name = project_in[PHOTOGRAMMETRY_AERIAL_DEVICE_NAME_INDEX]
        device_type = project_in[PHOTOGRAMMETRY_AERIAL_DEVICE_TYPE_INDEX]
        lat_top_left = project_in[PHOTOGRAMMETRY_AERIAL_LATITUDE_TOP_LEFT_INDEX]
        long_top_left = project_in[PHOTOGRAMMETRY_AERIAL_LONGITUDE_TOP_LEFT_INDEX]
        lat_bottom_right = project_in[PHOTOGRAMMETRY_AERIAL_LATITUDE_BOTTOM_RIGHT_INDEX]
        long_bottom_right = project_in[PHOTOGRAMMETRY_AERIAL_LONGITUDE_BOTTOM_RIGHT_INDEX]
        data_set_size = project_in[PHOTOGRAMMETRY_AERIAL_DATA_SIZE_INDEX]

        survey_data = add_data_set_data(survey_data, device_section_template,
                          data_set_section_template, device_name,
                          device_type, "Photogrammetry - Aerial", lat_top_left,
                          long_top_left, lat_bottom_right, long_bottom_right,
                          data_set_size, num_data_sets_added)
        num_data_sets_added += 1

    if project_in[DATA_DERIVATIVES_DSM_ORTHO_DEVICE_NAME_INDEX] is not None and \
       project_in[DATA_DERIVATIVES_DSM_ORTHO_DEVICE_NAME_INDEX] != "":
        device_name = project_in[DATA_DERIVATIVES_DSM_ORTHO_DEVICE_NAME_INDEX]
        device_type = project_in[DATA_DERIVATIVES_DSM_ORTHO_DEVICE_TYPE_INDEX]
        lat_top_left = project_in[DATA_DERIVATIVES_DSM_ORTHO_LATITUDE_TOP_LEFT_INDEX]
        long_top_left = project_in[DATA_DERIVATIVES_DSM_ORTHO_LONGITUDE_TOP_LEFT_INDEX]
        lat_bottom_right = project_in[DATA_DERIVATIVES_DSM_ORTHO_LATITUDE_BOTTOM_RIGHT_INDEX]
        long_bottom_right = project_in[DATA_DERIVATIVES_DSM_ORTHO_LONGITUDE_BOTTOM_RIGHT_INDEX]
        data_set_size = project_in[DATA_DERIVATIVES_DSM_ORTHO_DATA_SIZE_INDEX]

        survey_data = add_data_set_data(survey_data, device_section_template,
                          data_set_section_template, device_name,
                          device_type, "Data Derivatives - DSM/Ortho",
                          lat_top_left, long_top_left, lat_bottom_right,
                          long_bottom_right, data_set_size,
                          num_data_sets_added)
        num_data_sets_added += 1

    if project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DEVICE_NAME_INDEX] is not None and \
       project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DEVICE_NAME_INDEX] != "":
        device_name = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DEVICE_NAME_INDEX]
        device_type = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DEVICE_TYPE_INDEX]
        lat_top_left = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LATITUDE_TOP_LEFT_INDEX]
        long_top_left = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LONGITUDE_TOP_LEFT_INDEX]
        lat_bottom_right = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LATITUDE_BOTTOM_RIGHT_INDEX]
        long_bottom_right = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_LONGITUDE_BOTTOM_RIGHT_INDEX]
        data_set_size = project_in[DATA_DERIVATIVES_3D_PHOTOGRAMMETRY_DATA_SIZE_INDEX]

        survey_data = add_data_set_data(survey_data, device_section_template,
                          data_set_section_template, device_name,
                          device_type, "Data Derivatives - 3D photogrammetry",
                          lat_top_left, long_top_left, lat_bottom_right,
                          long_bottom_right, data_set_size, num_data_sets_added)
        num_data_sets_added += 1

    # organizations sections
    project_id = project_in[PROJECT_ID]
    organizations = entities_in[str(project_id)]
    for i in range(len(organizations)):
        organization_data = deepcopy(organization_section_template)
        organization = organizations[i]

        organization_data[sm.ORGANIZATION_NAME_OFFSET][sm.FILL_IN_COLUMN] = organization[ORGANIZATION_NAME_INDEX]
        organization_data[sm.ORGANIZATION_URL_OFFSET][sm.FILL_IN_COLUMN] = organization[ORGANIZATION_URL_INDEX]
        organization_data[sm.ENTITY_TYPE_OFFSET][sm.FILL_IN_COLUMN] = organization[ENTITY_TYPE_INDEX]

        organization_data_insert_index = sm.ORIGINAL_DEVICE_SECTION + \
                                         num_data_sets_added * (sm.DEVICE_SECTION_LENGTH +
                                                                sm.DATA_SET_SECTION_LENGTH) + \
                                         i * sm.ORGANIZATION_SECTION_LENGTH + \
                                         2

        survey_data = survey_data[:organization_data_insert_index] + \
                      organization_data + \
                      survey_data[organization_data_insert_index:]

    return survey_data


def add_data_set_data(survey_data_in: list[list[str]],
                      device_section_template_in: list[list[str]],
                      data_set_section_template_in: list[list[str]],
                      device_name_in: str, device_type_in: str,
                      data_set_type_in: str, lat_top_left_in: int,
                      long_top_left_in: int, lat_bottom_right_in: int,
                      long_bottom_right_in: int, data_set_size_in: int,
                      num_data_sets_added_in: int) -> list[list[str]]:
    device_section_data = deepcopy(device_section_template_in)
    device_section_data[sm.DEVICE_NAME_ROW_OFFSET][sm.FILL_IN_COLUMN] = device_name_in
    device_section_data[sm.DEVICE_TYPE_ROW_OFFSET][sm.FILL_IN_COLUMN] = device_type_in
    device_section_data[sm.DEVICE_DATA_TYPE_OFFSET][sm.FILL_IN_COLUMN] = data_set_type_in

    data_set_section_data = deepcopy(data_set_section_template_in)
    data_set_section_data[sm.DATA_SET_TYPE_OFFSET][sm.FILL_IN_COLUMN] = data_set_type_in
    data_set_section_data[sm.LATITUDE_TOP_LEFT_OFFSET][sm.FILL_IN_COLUMN] = lat_top_left_in
    data_set_section_data[sm.LONGITUDE_TOP_LEFT_OFFSET][sm.FILL_IN_COLUMN] = long_top_left_in
    data_set_section_data[sm.LATITUDE_BOTTOM_RIGHT_OFFSET][sm.FILL_IN_COLUMN] = lat_bottom_right_in
    data_set_section_data[sm.LONGITUDE_BOTTOM_RIGHT_OFFSET][sm.FILL_IN_COLUMN] = long_bottom_right_in
    data_set_section_data[sm.SIZE_OFFSET][sm.FILL_IN_COLUMN] = data_set_size_in

    device_section_insert_index = sm.ORIGINAL_DEVICE_SECTION + \
                                  sm.DEVICE_SECTION_LENGTH * num_data_sets_added_in
    survey_data_in = survey_data_in[:device_section_insert_index] + \
                     device_section_data + \
                     survey_data_in[device_section_insert_index:]

    data_set_section_insert_index = sm.ORIGINAL_DEVICE_SECTION + \
                                    sm.DEVICE_SECTION_LENGTH * (num_data_sets_added_in + 1) + \
                                    sm.DATA_SET_SECTION_LENGTH * num_data_sets_added_in + 1
    survey_data_in = survey_data_in[:data_set_section_insert_index] + \
                     data_set_section_data + \
                     survey_data_in[data_set_section_insert_index:]

    return survey_data_in

def get_survey_template_data() -> list[list[str]]:
    """
    Reads and parses the data from the survey template csv file to be used as a template.

    :return: A list of list of strings representing the survey template data
    """
    survey_template = open(SURVEY_TEMPLATE_PATH, 'r', encoding='Latin1')
    survey_template_reader = csv.reader(survey_template)

    survey_data_out = []
    for row in survey_template_reader:
        survey_data_out.append(row)

    survey_template.close()

    return survey_data_out

def create_filled_survey(filled_survey_data_in: list[list[str]], filled_survey_path_in: str) -> None:
    """
    Fills in some of the given survey data based on the given survey path.

    :param filled_survey_data_in: The list of list of strings representing the survey data to write
    :param filled_survey_path_in: The string path of the file to write to
    :return: None
    """
    filled_survey = open(filled_survey_path_in, 'w', encoding='Latin1', newline="")
    writer = csv.writer(filled_survey)

    for csv_row in filled_survey_data_in:
        writer.writerow(csv_row)

    filled_survey.close()

    print("Data saved to: " + filled_survey_path_in)

if __name__ == '__main__':
    if len(sys.argv) == 3:
        microsoft_access_file_path = sys.argv[1]
        new_survey_directory_path = sys.argv[2]
    else:
        print("Error: incorrect number of arguments.\n" +
              "Usage: python microsoft_access_parser.py [microsoft_access_file_path] " +
              "[new_survey_directory_path]")
        exit()

    cursor = make_cursor(microsoft_access_file_path)
    projects = parse_project_master(cursor)
    entities = parse_project_entities(cursor)

    process_projects(projects, entities, new_survey_directory_path)



