import csv
from datetime import datetime
import os
import sys
import time
import exifread
import platform

import PIL.Image
import PIL.ExifTags

import survey_metadata as sm

SURVEY_TEMPLATE_PATH = "CSV_files/OH_Metadata_Survey_Template.csv"

LATITUDE_KEY = "GPS GPSLatitude"
LONGITUDE_KEY = "GPS GPSLongitude"

MAKE_KEY = "Make"
MODEL_KEY = "Model"

LIDAR_TERRESTRIAL = "LiDAR - Terrestrial"
LIDAR_AERIAL = "LiDAR - Aerial"
PHOTOGRAMMETRY_TERRESTRIAL = "Photogrammetry - Terrestrial"
PHOTOGRAMMETRY_AERIAL = "Photogrammetry - Aerial"

model_to_camera_name_dict = {}
MODEL_TO_CAMERA_NAME_LIBRARY_PATH = "CSV_files/Model_To_Camera_Name_Library.csv"

camera_name_to_device_type_dict = {}
CAMERA_NAME_TO_DEVICE_TYPE_LIBRARY_PATH = "CSV_files/Camera_Name_To_Device_Type_Library.csv"


def get_survey_template_data() -> list[list[str]]:
    """
    Reads and parses the data from the survey template csv file to be used as a template.

    :return: A list of list of strings representing the survey template data
    """
    survey_template = open(SURVEY_TEMPLATE_PATH, 'r', encoding='Latin1')
    survey_template_reader = csv.reader(survey_template)

    survey_data_out = []
    for row in survey_template_reader:
        survey_data_out.append(row)

    survey_template.close()

    return survey_data_out


def load_dicts() -> None:
    """
    Loads the model to camera dictionary and the make and model to camera dictionary into memory from their respective
    csv files.

    :return: None
    """
    print("Loading Dictionaries:")

    # Load model to camera library
    print("Loading Model to Camera Dictionary from " + MODEL_TO_CAMERA_NAME_LIBRARY_PATH + ".")
    model_to_camera_library = open(MODEL_TO_CAMERA_NAME_LIBRARY_PATH, 'r', encoding='Latin1')
    model_to_camera_library_reader = csv.reader(model_to_camera_library)
    next(model_to_camera_library_reader)

    for row in model_to_camera_library_reader:
        if len(row) == 2:
            model_to_camera_name_dict[row[0]] = row[1]

    print("Model to Camera Dictionary Loaded.")

    # Load make and model to device type library
    print("Loading Make and Model to Device Type Dictionary from " + CAMERA_NAME_TO_DEVICE_TYPE_LIBRARY_PATH + ".")
    camera_name_to_device_type_library = open(CAMERA_NAME_TO_DEVICE_TYPE_LIBRARY_PATH, 'r', encoding='Latin1')
    camera_name_to_device_type_library_reader = csv.reader(camera_name_to_device_type_library)
    next(camera_name_to_device_type_library_reader)

    for row in camera_name_to_device_type_library_reader:
        if len(row) == 2:
            camera_name_to_device_type_dict[row[0]] = row[1]

    print("Make and Model to Camera Dictionary Loaded.")


def fill_survey_data(survey_data_in: list[list[str]], doi_data_folder_path_in: str) -> list[list[str]]:
    """
    Parses and retrieves the metadata from the given doi data folder path and uses it to fill out the given survey data.

    :param survey_data_in: The list of list of string representing the survey template data
    :param doi_data_folder_path_in: The string path of the doi data folder to read the metadata from
    :return: The list of list of strings representing the filled out data survey
    """
    folder_data = get_folder_data(doi_data_folder_path_in)

    survey_data_in[sm.START_DATE_ROW][sm.FILL_IN_COLUMN] = folder_data["start_date"]
    survey_data_in[sm.END_DATE_ROW][sm.FILL_IN_COLUMN] = folder_data["end_date"]

    # Device Sections
    # Look for start of device section
    for i in len(survey_data_in):
        if survey_data_in[i][0] == "deviceName":
            device_section_row = i
            break

    device_sections = folder_data["device_sections"]
    for i in range(len(device_sections)):
        base_row = device_section_row + i * sm.DEVICE_SECTION_LENGTH
        survey_data_in[base_row + sm.DEVICE_NAME_ROW_OFFSET][sm.FILL_IN_COLUMN] = device_sections[i]["camera_name"]
        # survey_data_in[base_row + sm.DEVICE_TYPE_ROW_OFFSET][sm.FILL_IN_COLUMN] = device_sections[i]["device_type"]
        survey_data_in[base_row + sm.DEVICE_DATA_TYPE_OFFSET][sm.FILL_IN_COLUMN] = device_sections[i]["data_type"]

    # Data Sets
    # Look for start of data set section
    for i in len(survey_data_in):
        if survey_data_in[i][0] == "dataType":
            data_set_row = i
            break

    data_sets = folder_data["data_sets"]
    for i in range(len(data_sets)):
        base_row = data_set_row + i * sm.DATA_SET_SECTION_LENGTH
        survey_data_in[base_row + sm.DATA_SET_TYPE_OFFSET][sm.FILL_IN_COLUMN] = data_sets[i]["data_type"]
        survey_data_in[base_row + sm.LATITUDE_TOP_LEFT_OFFSET][sm.FILL_IN_COLUMN] = data_sets[i]["latitude_top_left"]
        survey_data_in[base_row + sm.LONGITUDE_TOP_LEFT_OFFSET][sm.FILL_IN_COLUMN] = data_sets[i]["longitude_top_left"]
        survey_data_in[base_row + sm.LATITUDE_BOTTOM_RIGHT_OFFSET][sm.FILL_IN_COLUMN] = data_sets[i]["latitude_bottom_right"]
        survey_data_in[base_row + sm.LONGITUDE_BOTTOM_RIGHT_OFFSET][sm.FILL_IN_COLUMN] = data_sets[i]["longitude_bottom_right"]
        survey_data_in[base_row + sm.SIZE_OFFSET][sm.FILL_IN_COLUMN] = data_sets[i]["size"]

    filled_survey_data_out = survey_data_in

    return filled_survey_data_out


def get_folder_data(doi_data_folder_path_in: str) -> \
    dict[str, str | int | list[dict[str, str | int | list[...]]] | list[dict[str, str | int]]]:
    """
    Reads the given folder's metadata and returns a dict with all the metadata.
    :param doi_data_folder_path_in: The doi data folder to get metadata from

    :return: A dict with all of the metadata in the following format: dict["start_date": str, "end_date": str,
             "device_sections": list[dict["data_type": str, "camera_name": str], "data_sets": list["data_type": str,
             "latitude_top_left": str, "longitude_top_left": str, "latitude_bottom_right": str,
             longitude_bottom_right": str]
    """
    start_date = time.time()
    end_date = 0

    directory_count = 0

    folder_data = {}

    device_sections = []
    data_sets = []
    for directory_name in os.listdir(doi_data_folder_path_in):
        directory_path = doi_data_folder_path_in + "/" + directory_name

        max_dir_latitude = -90
        max_dir_longitude = -180
        min_dir_latitude = 90
        min_dir_longitude = 180

        device_selection_data = {}
        data_set_data = {}
        files = os.listdir(directory_path)
        for filename in (files[0:1] + files[-1:]):
            path_to_file = directory_path + "/" + filename
            f = open(path_to_file, 'rb')

            # Get latitude/longitude values
            tags = exifread.process_file(f)
            latitude = str(tags[LATITUDE_KEY])[1:-1].split(',')
            longitude = str(tags[LONGITUDE_KEY])[1:-1].split(',')
            latitude = dms_to_dd(latitude[0], latitude[1], latitude[2])
            longitude = dms_to_dd(longitude[0], longitude[1], longitude[2])

            max_dir_latitude = max(max_dir_latitude, latitude)
            max_dir_longitude = max(max_dir_longitude, longitude)
            min_dir_latitude = min(min_dir_latitude, latitude)
            min_dir_longitude = min(min_dir_longitude, longitude)

            # Get creation time, modification time, and file size
            stats = os.stat(path_to_file)

            creation_timestamp = ""
            if platform.system() == 'Windows':
                creation_timestamp = stats.st_ctime
            else:
                try:
                    creation_timestamp = stats.st_birthtime
                except AttributeError:
                    # We're probably on Linux. No easy way to get creation dates here,
                    # so we'll settle for when its content was last modified.
                    creation_timestamp = stats.st_mtime

            start_date = min(start_date, creation_timestamp)
            end_date = max(end_date, stats.st_mtime)

            size = sizeof_fmt(stats.st_size)
            data_set_data["size"] = size

            # Get make and model values
            img = PIL.Image.open(path_to_file)
            exif = {
                PIL.ExifTags.TAGS[k]: v
                for k, v in img.getexif().items()
                if k in PIL.ExifTags.TAGS
            }

            make = exif[MAKE_KEY]
            model = exif[MODEL_KEY]

        data_type = ""
        if "lidar_terrestrial" in doi_data_folder_path_in:
            data_type = LIDAR_TERRESTRIAL
        elif "lidar_aerial" in doi_data_folder_path_in:
            data_type = LIDAR_AERIAL
        elif "photogrammetry_terrestrial" in doi_data_folder_path_in:
            data_type = PHOTOGRAMMETRY_TERRESTRIAL
        elif "photogrammetry_aerial" in doi_data_folder_path_in:
            data_type = PHOTOGRAMMETRY_AERIAL
        device_selection_data["data_type"] = data_type
        data_set_data["data_type"] = data_type

        try:
            device_selection_data["camera_name"] = make + " " + model_to_camera_name_dict[model]

            try:
                device_selection_data["device_type"] = camera_name_to_device_type_dict[
                    device_selection_data["camera_name"]]
            except KeyError:
                print("The make and model: " + device_selection_data["camera_name"] + "is not in the make and model "
                      "to device type dictionary: " + CAMERA_NAME_TO_DEVICE_TYPE_LIBRARY_PATH)
                device_selection_data["device_type"] = ""

        except KeyError:
            print("The model: " + model + " is not in the model to camera dictionary: " + MODEL_TO_CAMERA_NAME_LIBRARY_PATH)
            device_selection_data["camera_name"] = ""

        data_set_data["latitude_top_left"] = max_dir_latitude
        data_set_data["longitude_top_left"] = max_dir_longitude
        data_set_data["latitude_bottom_right"] = min_dir_latitude
        data_set_data["longitude_bottom_right"] = min_dir_longitude

        directory_count += 1

        device_sections.append(device_selection_data)
        data_sets.append(data_set_data)

    start_date = datetime.fromtimestamp(start_date).strftime('%Y-%m-%d %H:%M:%S')
    end_date = datetime.fromtimestamp(end_date).strftime('%Y-%m-%d %H:%M:%S')

    folder_data["start_date"] = start_date
    folder_data["end_date"] = end_date

    folder_data["device_sections"] = device_sections
    folder_data["data_sets"] = data_sets

    return folder_data


def dms_to_dd(d: int, m: int, s: int) -> float:
    """
    Takes a latitude or longitude coordinate in degrees, minutes, and seconds format and converts it to decimal degrees
    format.

    :param d: The degrees of the coordinate
    :param m: The minutes of the coordinate
    :param s: The seconds of the coordinate
    :return: The coordinate in decimal degrees format
    """
    if '/' in m:
        m = m.split('/')
        m = float(m[0]) / float(m[1])

    if d[0] == '-':
        dd = float(d) - float(m)/60 - float(s)/3600
    else:
        dd = float(d) + float(m)/60 + float(s)/3600
    return dd


# https://stackoverflow.com/questions/1094841/get-human-readable-version-of-file-size
def sizeof_fmt(num: float, suffix: str = "B") -> str:
    """
    Returns a string representing a number of bytes in an easy to read format by using higher powers of bytes.

    :param num: The number of bytes to format
    :param suffix: The suffix to use at the end of the string. The default suffix is "B"
    :return: The string representation of the number of bytes
    """
    for unit in ["", "Ki", "Mi", "Gi", "Ti", "Pi", "Ei", "Zi"]:
        if abs(num) < 1024.0:
            return f"{num:3.1f}{unit}{suffix}"
        num /= 1024.0
    return f"{num:.1f}Yi{suffix}"


def create_filled_survey(filled_survey_data_in: list[list[str]], filled_survey_path_in: str) -> None:
    """
    Fills in some of the given survey data based on the given survey path.

    :param filled_survey_data_in: The list of list of strings representing the survey data to write
    :param filled_survey_path_in: The string path of the file to write to
    :return: None
    """
    filled_survey = open(filled_survey_path_in, 'w', encoding='Latin1', newline="")
    writer = csv.writer(filled_survey)

    for csv_row in filled_survey_data_in:
        writer.writerow(csv_row)

    filled_survey.close()


if __name__ == '__main__':
    if len(sys.argv) == 3:
        doi_data_folder_path = sys.argv[1]
        filled_survey_path = sys.argv[2]
    else:
        print("Error: incorrect number of arguments.\n" +
              "Usage: python OH_Metadata_Survey_Filler.py [survey_template_path] " +
              "[doi_data_folder_path] [filled_survey_name]")
        exit()

    load_dicts()
    survey_data = get_survey_template_data()
    filled_survey_data = fill_survey_data(survey_data, doi_data_folder_path)
    create_filled_survey(filled_survey_data, filled_survey_path)
