import pandas as pd
import requests
import csv


PROJECTS_MASTER_CSV_PATH = r'CSV_files/Projects_MASTER.csv'

def read_dois() -> list[str]:
    """
    Returns a list of string doi's from the Projects Master csv file

    :return: A list of string doi's from the Projects Master csv file
    """
    data = pd.read_csv(PROJECTS_MASTER_CSV_PATH)
    dois = data['doi'].T.values.tolist()

    return dois


def get_invalid_dois(dois_list: list[str]) -> list[str]:
    """
    Returns a list of invalid dois from the given dois list.

    :param dois_list: The list of dois to check validity
    :return: A list of invalid dois from the given dois list
    """
    invalid_dois = []
    for doi in dois_list:
        if not is_valid_doi(doi):
            invalid_dois.append(doi)

    return invalid_dois


def is_valid_doi(doi: str) -> bool:
    """
    Checks whether the doi is valid. Makes a request to doi.org to see if the doi link is working. Checks if the request
    timed out or if the status code is not success.

    :param doi: The doi to check
    :return: whether the doi is valid
    """
    try:
        request = requests.get("https://doi.org/10.26301/" + doi)
    except requests.exceptions.Timeout:
        return False

    return request.status_code != 404


def write_to_file(file_path: str, invalid_dois_list: list[str]) -> None:
    """
    Writes the list of invalid dois to the given file path. Creates the file if it does not exist.

    :param file_path: The string file path to write to
    :param invalid_dois_list: The list of invalid dois to write
    :return: None
    """
    f = open(file_path, 'w', newline="")
    writer = csv.writer(f)

    writer.writerow(["doi"])
    for doi in invalid_dois_list:
        writer.writerow([doi])

    f.close()


if __name__ == '__main__':
    final_invalid_dois = get_invalid_dois(read_dois())
    print(final_invalid_dois)
    print("There are " + str(len(final_invalid_dois)) + " invalid dois")
    write_to_file("Output/invalid_dois.csv", final_invalid_dois)
