import csv
import sys

import requests
import base64
import xml_generator
import database_parser
import json

GET_URL = "https://api.datacite.org/dois/10.26301/"
POST_URL = "https://api.datacite.org/dois"
REQUEST_HEADERS = {"Content-Type": "application/vnd.api+json"}
DATA_CITE_AUTH = ('CYARK.OH', 'm3s4v3rd3')

ORGANIZATION_NAME_INDEX = 0
ORGANIZATION_URL_INDEX = 1
IS_AUTHORITY_INDEX = 2
IS_COLLECTOR_INDEX = 3
IS_FUNDER_INDEX = 4
IS_PARTNER_INDEX = 5
IS_CONTRIBUTOR_INDEX = 6

DOI_INDEX = 0
PROJECT_NAME_INDEX = 1
COUNTRY_INDEX = 2
LATITUDE_INDEX = 3
LONGITUDE_INDEX = 4
STATUS_INDEX = 5
PROJECT_DESCRIPTION_INDEX = 6
SITE_DESCRIPTION_INDEX = 7
COLLECTION_DATE_START_INDEX = 8
COLLECTION_DATE_END_INDEX = 9
PUBLISH_DATE_INDEX = 10
LICENSE_TYPE_INDEX = 11
LICENSE_LINK_INDEX = 12
EXTERNAL_PROJECT_LINK_INDEX = 13
ADDITIONAL_INFO_LINK_INDEX = 14
KEYWORDS_INDEX = 15
POTREE_VIEWER_INDEX = 16

UPCOMING_STATUS = "Upcoming"
PUBLISHED_STATUS = "Published"

NEW_DOI_JSON = {
    "data": {
        "type": "dois",
        "attributes": {
            "prefix": "10.26301",
        }
    }
}


def upload_doi_draft(doi_csv_path: str) -> bool:
    """
    Creates a DOI for the doi project on Datacite, adds the doi ID to the given doi csv, and then finally uploads the
    doi csv data to Datacite.

    :param doi_csv_path: The doi csv string to be uploaded
    :return: Whether the upload was successful
    """
    # open csv file
    doi_csv = open(doi_csv_path, 'r', encoding='Latin1')
    doi_reader = csv.reader(doi_csv)

    # parse values from file
    values = []
    for row in doi_reader:
        values.append(row[xml_generator.VALUE_COL_INDEX])

    doi = values[2]

    status = values[xml_generator.STATUS_ROW_INDEX]
    project_description = values[xml_generator.PROJECT_DESCRIPTION_ROW_INDEX]
    site_description = values[xml_generator.SITE_DESCRIPTION_ROW_INDEX]
    update_descriptions_to_csv(xml_generator.remove_tags_from_description(project_description),
                               xml_generator.remove_tags_from_description(site_description), doi_csv_path)

    xml_str = xml_generator.generate_xml_from_csv(doi_csv_path)
    encoded_xml = base64.b64encode(xml_str.encode("ascii", "replace"))
    data = json.dumps(generate_json(doi, encoded_xml, status))

    # check if doi exists
    doi_response = requests.get(GET_URL + doi, headers=REQUEST_HEADERS, auth=DATA_CITE_AUTH)

    if doi != "" and doi_response.status_code != 404:
        r = requests.put(GET_URL + "/" + doi, headers=REQUEST_HEADERS, data=str(data),
                         auth=DATA_CITE_AUTH)
        doi_csv.close()
        if r.status_code == 200:
            print("Successfully updated doi: " + doi)
            return True
        else:
            print("Error: could not update doi: " + doi)
            print(r.content)
            print("Status Code: " + str(r.status_code))
            return False
    else:
        post_response = requests.post(POST_URL, headers=REQUEST_HEADERS, data=str(json.dumps(NEW_DOI_JSON)),
                                      auth=DATA_CITE_AUTH)
        if post_response.status_code == 201:
            doi = post_response.json()["data"]["attributes"]["suffix"]

            add_doi_to_csv(doi, doi_csv_path)

            xml_str = xml_generator.generate_xml_from_csv(doi_csv_path)
            encoded_xml = base64.b64encode(xml_str.encode("ascii", "replace"))
            data = json.dumps(generate_json(doi, encoded_xml, status))

            doi_csv.close()

            print("Successfully created a draft for doi: " + doi)
            print(post_response.json())

            put_response = requests.put(GET_URL + "/" + doi, headers=REQUEST_HEADERS, data=str(data),
                                        auth=DATA_CITE_AUTH)
            if put_response.status_code == 200:
                print("Successfully updated doi: " + doi)
                return True
            else:
                print("Error: could not update doi: " + doi)
                print(put_response.content)
                print("Status Code: " + str(put_response.status_code))
                return False
        else:
            doi_csv.close()
            print("Error: could not create a draft for doi: " + "")
            print(post_response.content)
            print("Status Code: " + str(post_response.status_code))
            return False


def add_doi_to_csv(doi: str, doi_csv_path: str) -> None:
    """
    Adds the given doi string to the given doi csv, in the location, specifically for the doi string.

    :param doi: The doi to be written
    :param doi_csv_path: The path of doi csv to be written to
    :return: None
    """

    doi_csv = open(doi_csv_path, 'r', encoding='Latin1')
    doi_reader = csv.reader(doi_csv)

    csv_data = []
    for row in doi_reader:
        csv_data.append(row)

    csv_data[2][3] = doi
    doi_csv.close()

    doi_csv = open(doi_csv_path, 'w', encoding='Latin1', newline="")
    writer = csv.writer(doi_csv)

    for csv_row in csv_data:
        writer.writerow(csv_row)

    doi_csv.close()

    print("Updated DOI: " + doi_csv_path)


def update_descriptions_to_csv(project_description: str, site_description: str, doi_csv_path: str) -> None:
    """
    Replaces the project description and site description fields in given doi csv, with the given descriptions.

    :param project_description: The project description to replace the project description in the doi csv
    :param site_description: The site description to replace the site description in the doi csv
    :param doi_csv_path: The path of the doi csv to update
    :return: None
    """
    doi_csv = open(doi_csv_path, 'r', encoding='Latin1')
    doi_reader = csv.reader(doi_csv)

    csv_data = []
    for row in doi_reader:
        csv_data.append(row)

    csv_data[8][3] = project_description
    csv_data[9][3] = site_description

    doi_csv.close()

    doi_csv = open(doi_csv_path, 'w', encoding='Latin1', newline="")
    writer = csv.writer(doi_csv)

    for csv_row in csv_data:
        writer.writerow(csv_row)

    doi_csv.close()

    print("Updated Description: " + doi_csv_path)


def upload_doi_drafts(project_master_path: str, project_entities_path: str, organizations_path: str) -> None:
    """
    Using the data from the project master csv, project entities csv, and organizations csv files, create DOIs on
    Datacite and upload all the doi information.

    :param project_master_path: The path to the project master csv
    :param project_entities_path: The poth to the project entities csv
    :param organizations_path: The path to the organization csv
    :return: None
    """

    doi_to_ids_dict = database_parser.get_doi_to_ids_dict(project_entities_path)
    id_to_organization_dict = database_parser.get_id_to_organization_dict(organizations_path)
    doi_to_organizations_dict = database_parser.get_doi_to_organizations_dict(doi_to_ids_dict, id_to_organization_dict)

    projects_master_reader = database_parser.get_projects_master_reader(project_master_path)
    header = next(projects_master_reader)

    # Check file as empty
    if header is not None:
        dois = doi_to_organizations_dict.keys()
        for project in projects_master_reader:
            doi = project[DOI_INDEX]

            organizations = []
            if doi in dois:
                organizations = doi_to_organizations_dict[doi]

            xml_str = xml_generator.generate_xml(project, organizations)
            encoded_xml = base64.b64encode(xml_str.encode("ascii", "replace"))
            data = json.dumps(generate_json(doi, encoded_xml, project[STATUS_INDEX]))

            doi_response = requests.get(GET_URL + doi,
                                        headers=REQUEST_HEADERS, auth=DATA_CITE_AUTH)

            if doi_response.status_code != 404:
                r = requests.put(GET_URL + "/" + doi, headers=REQUEST_HEADERS, data=str(data),
                                 auth=DATA_CITE_AUTH)
                if r.status_code == 200:
                    print("Successfully updated doi: " + project[DOI_INDEX])
                else:
                    print("Error: could not update doi: " + project[DOI_INDEX])
                    print(r.content)
                    print("Status Code: " + str(r.status_code))
            else:
                r = requests.post(POST_URL, headers=REQUEST_HEADERS, data=str(data), auth=DATA_CITE_AUTH)
                if r.status_code == 201:
                    print("Successfully created a draft for doi: " + project[DOI_INDEX])
                else:
                    print("Error: could not create a draft for doi: " + project[DOI_INDEX])
                    print(r.content)
                    print("Status Code: " + str(r.status_code))


def generate_json(doi: str, encoded_xml: bytes, project_status: str) \
                  -> dict[str, dict[str, str | dict[str, str]]]:
    """
    Generates a json for use in uploading doi data to Datacite, using the given doi string, encoded xml data, and
    project status.

    :param doi: The doi of the project of the json
    :param encoded_xml: An ascii encoding, with replacement for unknown characters, of the xml form of the doi data
    :param project_status: The status of the project
    :return: A dict representing the json to be uploaded to Datacite, which contains the data of the doi project
    """

    event = "draft"

    if project_status == PUBLISHED_STATUS:
        event = "publish"

    project_json = {
        "data": {
            "id": "10.26301/" + doi,
            "type": "dois",
            "attributes": {
                "event": event,
                "doi": "10.26301/" + doi,
                "url": "https://openheritage3d.org/project/" + doi,
                "xml": str(encoded_xml)[1:]
            }
        }
    }

    return project_json


if __name__ == '__main__':
    if len(sys.argv) == 2:
        csv_path = sys.argv[1]
        upload_doi_draft(csv_path)
    else:
        upload_doi_drafts("CSV_files/Projects_MASTER.csv", "CSV_files/Project_Entities.csv", "CSV_files/Organizations.csv")
