import csv

DOI_PROJECT_INDEX = 0
ID_ORGANIZATION_INDEX = 2
IS_CONTRIBUTOR_INDEX = 7

ID_INDEX = 0
ORGANIZATION_NAME_INDEX = 1
ORGANIZATION_URL_INDEX = 2
SITE_AUTHORITY_INDEX = 3
COLLECTOR_INDEX = 4
FUNDER_INDEX = 5
PARTNER_INDEX = 6
CONTRIBUTOR_INDEX = 7

DATASETS_DOI_PROJECT_INDEX = 1
DATA_TYPE_INDEX = 3


def get_projects_master_reader(project_master_path: str) -> csv.reader:
    """
    Returns a csv reader for the given project master path. The file is opened is read only mode, with a Latin1
    encoding.

    :param project_master_path: The file path to the project master csv file
    :return: The csv reader for the project master path
    """
    projects_master = open(project_master_path, 'r', encoding='Latin1')
    x = csv.reader(projects_master)

    return x


def get_doi_to_ids_dict(project_entities_path: str) -> dict[str, list[list[str]]]:
    """
    Parses the project entities file and returns a dict, where each key is a doi which points to a list ids of
    organizations that contributed to the doi project, along with some data of the organization's contributions.

    :param project_entities_path: The project entities csv file path
    :return: A dict, where each key is doi string and each value is a list of organization related lists of strings. The
             list of strings have the structure of, [id_Organization, isAuthority (0 for false, 1 for true), isCollector
             (0 = false, 1 = true), isFunder (0 = false, 1 = true), isPartner (0 = false, 1 = true), isContributor (0 =
             false, 1 = true)]
    """
    doi_to_ids_dict = {}
    with open(project_entities_path, 'r', encoding='Latin1') as projects_entities:
        projects_entities_reader = csv.reader(projects_entities)
        header = next(projects_entities_reader)

        # Check file as empty
        if header is not None:
            for entity in projects_entities_reader:
                dois = doi_to_ids_dict.keys()
                doi = entity[DOI_PROJECT_INDEX]

                if doi in dois:
                    if entity[IS_CONTRIBUTOR_INDEX] == "1":
                        doi_to_ids_dict[doi].insert(0, entity[ID_ORGANIZATION_INDEX:])
                    else:
                        doi_to_ids_dict[doi].append(entity[ID_ORGANIZATION_INDEX:])
                else:
                    doi_to_ids_dict[doi] = [entity[ID_ORGANIZATION_INDEX:]]

    projects_entities.close()

    return doi_to_ids_dict


def get_id_to_organization_dict(organizations_path: str) -> dict[str, list[str]]:
    """
    Parses the organizations file and returns a dict, where each key is the id of an organization which points to a list
    of strings representing the data of the organization.

    :param organizations_path: The organizations csv file path
    :return: A dict, where each key is an id string of an organization and each value is a list of strings representing
             the organization. The structure of the list of strings is as follows: [organizationName, organizationURL,
             Site_Authority (0 = false, 1 = true), Collector (0 = false, 1 = true), Funder (0 = false, 1 = true),
             Partner (0 = false, 1 = true), Contributor (0 = false, 1 = true)]
    """
    id_to_organization_dict = {}
    with open(organizations_path, 'r', encoding='Latin1') as organizations:
        organizations_reader = csv.reader(organizations)
        header = next(organizations_reader)

        # Check file as empty
        if header is not None:
            for organization in organizations_reader:
                organization_id = organization[ID_INDEX]
                id_to_organization_dict[organization_id] = \
                    organization[ORGANIZATION_NAME_INDEX:ORGANIZATION_URL_INDEX + 1]

    organizations.close()

    return id_to_organization_dict


def get_doi_to_organizations_dict(ids_dict: dict[str, list[list[str]]], organization_dict: dict[str, list[str]]) \
                                  -> dict[str, list[list[str]]]:
    """
    Returns a dict, where each doi is linked to a list of organization related list of strings, which includes
    information on the organization's contribution to the linked doi and the organization's general information.
    :param ids_dict: A dict linking dois to list of organization data
    :param organization_dict: A dict linking organization ids to organization information.
    :return: A dict, where each key is a doi and each value is a list of organization related data, which is a list of
             strings. The list of strings is in the following format: [organizationName, organizationURL,
             Site_Authority (0 = false, 1 = true), Collector (0 = false, 1 = true), Funder (0 = false, 1 = true),
             Partner (0 = false, 1 = true), Contributor (0 = false, 1 = true), DOIAuthority (0 for false, 1 for true),
             DOICollector (0 = false, 1 = true), DOIFunder (0 = false, 1 = true), DOIPartner (0 = false, 1 = true),
             DOIContributor (0 = false, 1 = true)]
    """
    doi_to_organizations_dict = {}
    dois = ids_dict.keys()
    for doi in dois:
        entities = ids_dict[doi]
        doi_to_organizations_dict[doi] = []

        for entity in entities:
            entity_id = entity[0]
            if entity_id in organization_dict.keys():
                organization = organization_dict[entity_id]
                doi_to_organizations_dict[doi].append(organization + entity[1:])

    return doi_to_organizations_dict


def get_doi_to_datasets_dict(datasets_path: str) -> dict[str, list[list[str]]]:
    """
    Parses the datasets file and returns a dict linking dois to lists of dataset information, which a list of strings.

    :param datasets_path: The datasets csv file path
    :return: A dict, where each key is a doi string and each value is a list of dataset related data, which is a list
             of strings in the following format: [dataType, derivativeType, latitude_top_left, longitude_top_left,
             latitude_bottom_right, longitude_bottom_right, dataSize]
    """
    doi_to_datasets_dict = {}
    with open(datasets_path, 'r', encoding='Latin1') as datasets:
        datasets_reader = csv.reader(datasets)
        header = next(datasets_reader)

        # Check file as empty
        if header is not None:
            for dataset in datasets_reader:
                dois = doi_to_datasets_dict.keys()
                doi = dataset[DATASETS_DOI_PROJECT_INDEX]

                if doi in dois:
                    doi_to_datasets_dict[doi].append(dataset[DATA_TYPE_INDEX:])
                else:
                    doi_to_datasets_dict[doi] = [dataset[DATA_TYPE_INDEX:]]

    datasets.close()

    return doi_to_datasets_dict
