using UnityEngine;
using TMPro;

public class QuadClickHandler : MonoBehaviour
{
    public GameObject infoPanelPrefab; // Assign in Inspector
    private GameObject currentPanel;

    void Update()
    {
        if (Input.GetMouseButtonDown(0))
        {
            Ray ray = Camera.main.ScreenPointToRay(Input.mousePosition);
            if (Physics.Raycast(ray, out RaycastHit hit))
            {
                if (hit.transform.CompareTag("Quad"))
                {
                    ShowInfoPanel(hit.transform, hit.point);
                    return;
                }
            }

            // Clicked but hit nothing or something not tagged "Quad"
            if (currentPanel != null)
            {
                Destroy(currentPanel);
                currentPanel = null;
            }
        }
    }


    void ShowInfoPanel(Transform quadTransform, Vector3 hitPoint)
    {
        if (currentPanel != null) Destroy(currentPanel); // Only one panel at a time

        currentPanel = Instantiate(infoPanelPrefab);

        // Position panel slightly above the clicked point
        
        /*Vector3 offset = quadTransform.right * 0.1f + quadTransform.up * 0.05f;
        currentPanel.transform.position = hitPoint + offset;
        */

        Vector3 initialOffset = quadTransform.right * 0.005f + quadTransform.up * 0.005f;
        Vector3 roughPosition = hitPoint + initialOffset;

        Vector3 safePosition = ComputeSafePanelPosition(quadTransform, roughPosition);
        currentPanel.transform.position = safePosition;


        /*
        // Face the camera
        Vector3 cameraForward = Camera.main.transform.forward;
        cameraForward.y = -90; // flatten to horizontal plane
        cameraForward.x = 0;
        cameraForward.Normalize();

        // Set rotation so the panel faces camera direction but stays upright
        currentPanel.transform.rotation = Quaternion.LookRotation(cameraForward, Vector3.up);
        */

        // Set text
        var tmp = currentPanel.GetComponentInChildren<TextMeshProUGUI>();
        if (tmp != null)
        {
            QuadInfo info = quadTransform.GetComponent<QuadInfo>();
            if (info != null)
                tmp.text = info.displayText;
            else
                tmp.text = "No info available.";
        }

        // Set line AFTER the panel is positioned
        var lr = currentPanel.GetComponentInChildren<LineRenderer>();
        if (lr != null)
        {
            lr.positionCount = 2;

            // Point 0 = center of quad
            lr.SetPosition(0, quadTransform.position);

            // Point 1 = center of the panel
            lr.SetPosition(1, currentPanel.transform.position);
        }
    }

    Vector3 ComputeSafePanelPosition(Transform quadTransform, Vector3 basePosition)
    {
        Vector3 panelPosition = basePosition;
        float nudgeDistance = 0.01f;
        int maxAttempts = 10;

        for (int i = 0; i < maxAttempts; i++)
        {
            Collider[] nearby = Physics.OverlapSphere(panelPosition, 0.06f);
            bool overlapsQuad = false;

            foreach (Collider col in nearby)
            {
                if (col.CompareTag("Quad") && col.transform != quadTransform)
                {
                    overlapsQuad = true;
                    break;
                }
            }

            if (!overlapsQuad) break;

            // Nudge panel away from clicked quad's center
            Vector3 away = (panelPosition - quadTransform.position).normalized;
            panelPosition += away * nudgeDistance;
        }

        return panelPosition;
    }
}
